package com.flybits.android.push.models

/**
 * Indicates the action associated to the Push/MQTT notification.
 *
 * @param key the [String] value representing each [PushAction] option.
 */
enum class PushAction(val key: String) {

    /**
     * Indicates that a new entity was added. For example, new content is available to the user due
     * to a `Rule` status change.
     */
    ADDED("created"),

    /**
     * Indicates that a Push/MQTT notification was delivered with a custom action.
     */
    CUSTOM("custom"),

    /**
     * Indicates that an entity was removed. For example, content that use to be available to the
     * user should no longer be available due to a change in context associated to a `Rule`.
     */
    REMOVED("deleted"),

    /**
     * Indicates that a `Rule` has been updated.
     */
    @Deprecated("No longer supported")
    RULE_UPDATED("ruleUpdated"),

    /**
     * Indicates that the state of a `Rule` based on changing contextual information has been
     * changed from true to false or vice-versa.
     */
    @Deprecated("No longer supported")
    STATUS_CHANGED("statusChanged"),

    /**
     * Indicates that a unknown action was received with the specified entity.
     */
    UNKNOWN(""),

    /**
     * Indicates that an entity has been updated. For example if information associated to content
     * that is currently available to a user has been changed.
     */
    UPDATED("updated");

    companion object {

        /**
         * Get the [PushAction] enum value corresponding to an String representation.
         *
         * @param key the String representation of the [PushAction] enum.
         *
         * @return The [PushAction] enum for the String representation.
         */
        @JvmStatic
        fun fromKey(key: String?): PushAction {
            if (key == null) {
                return UNKNOWN
            }

            for (type in values()) {
                if (type.key.equals(key, ignoreCase = true)) {
                    return type
                }
            }
            return UNKNOWN
        }
    }
}
