package com.flybits.android.push

import android.content.Context
import android.os.Handler
import android.os.Looper
import com.flybits.android.push.api.FlyPushParsing
import com.flybits.android.push.deserializations.DeserializePushToken
import com.flybits.android.push.exceptions.FlybitsPushException
import com.flybits.android.push.models.Push
import com.flybits.android.push.models.PushToken
import com.flybits.android.push.models.PushToken.API
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.BasicResult
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.deserializations.IDeserializer
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Result
import com.google.firebase.iid.FirebaseInstanceId
import java.util.*
import java.util.concurrent.Executors

/**
 * The `PushManager` is responsible for all Push related mechanisms such as registering for
 * push notifications, parsing push notifications, and communicating with the FCM servers.
 */
object PushManager {

    /**
     * Disables Flybits-based push notifications including FCM and foreground notifications. This
     * process will remove the FCM token from the Flybits Push service as well as stopping the
     * foreground Push Service that is internal to the SDK.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     * @return The network request object that is triggers the `callback` based on the network
     * response.
     */
    @JvmStatic
    fun disablePush(context: Context, callback: BasicResultCallback? = null): BasicResult {
        return disablePush(context, SharedElementsFactory.get(context).getSavedJWTToken(), callback)
    }

    /**
     * Enable Flybits-based push notifications using FCM. This process will retrieve the FCM token
     * from Google and pass it to the Flybits Push Service.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param properties The list of key-value properties that should be added to the Push Token.
     * Default value is null.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     */
    @JvmStatic
    fun enableFCMPush(context: Context, properties: HashMap<String, String>?, callback: BasicResultCallback? = null) {
        FirebaseInstanceId.getInstance().instanceId.addOnSuccessListener { instanceIdResult ->
            val newToken = instanceIdResult.token
            enablePush(context, newToken, properties, callback)
        }
    }

    /**
     * Enable Flybits-based push notifications. This process will use the `token` and pass it
     * to the Flybits Push Service. This `token` can be an FCM token or any other one that is
     * necessary to register to push notifications.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param token The push token that should be saved for the logged in user.
     * @param properties The list of key-value properties that should be added to the Push Token.
     * Default value is null
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     * @param handler The handler that the callback methods will be invoked on.
     * Default value is a handler executing on the UI thread.
     */
    @JvmStatic
    fun enablePush(context: Context, token: String, properties: HashMap<String, String>?, callback: BasicResultCallback? = null
                   , handler: Handler = Handler(Looper.getMainLooper())): BasicResult {
        val executorService = Executors.newSingleThreadExecutor()
        val resultObject = BasicResult(callback, handler, executorService)
        executorService.execute {
            try {
                val resultSerializationObj = PushToken(token, properties)
                val deserialization = DeserializePushToken()
                val json = deserialization.toJson(resultSerializationObj)
                val pushResult: Result<Any> = FlyAway.post(context, API, json, null, "PushManager.enablePush", null)
                resultObject.setResult(pushResult)
            } catch (e: FlybitsException) {
                resultObject.setFailed(e)
            }
        }
        return resultObject
    }

    /**
     * Parse a FCM [Push] notification that is received from the Flybits platform. This push
     * notification is provided in a specific Flybits format which the SDK can parse. If the push
     * notification is not a Flybits notification and therefore cannot be parsed a
     * [FlybitsPushException] will be thrown within the
     * `PendingRequest.RequestCallbackWithResponse#onException(FlybitsException)`.
     *
     * @param mContext The [Context] of the application.
     * @param map The [Map] which contains the properties sent from the Flybits
     * platform. This should not be tempered with and should be sent directly to this
     * method from the FCM receiver.
     * @param callback The callback that initiated when the request is completed. It will contain
     * either a successful method or failure with a [FlybitsException] which
     * indicates the reason for failure. Default value is null.
     * @param <T> The `body` of the custom fields option. Some push notification may
     * provide additional fields with the request in order provide mechanisms for
     * deep-linking hence a unique structure can be provided.
    </T> */
    @JvmStatic
    fun <T : IDeserializer<*>> parsePushNotification(mContext: Context, map: Map<*, *>, callback: ObjectResultCallback<Push>? = null): Push? {
        try {
            val push = FlyPushParsing.parsePushNotification(mContext, map)
            callback?.onSuccess(push)
            return push
        } catch (e: FlybitsPushException) {
            callback?.onException(e)
        }
        return null
    }

    /**
     * Disables Flybits-based push notifications including FCM and foreground notifications. This
     * process will remove the FCM token from the Flybits Push service as well as stopping the
     * foreground Push Service that is internal to the SDK.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param jwttoken The jwt token associated with the user that the push notifications are being
     * disabled for.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     * @param handler The handler that the callback methods will be invoked on.
     * @return The network request object that is triggers the `callback` based on the network
     * response. Default value is a handler executing on the UI thread.
     */
    @JvmStatic
    fun disablePush(context: Context, jwttoken: String, callback: BasicResultCallback? = null
                    , handler: Handler = Handler(Looper.getMainLooper())): BasicResult {

        val executorService = Executors.newSingleThreadExecutor()
        val resultObject = BasicResult(callback, handler, executorService)
        executorService.execute {
            try {
                val mapOfHeaders = HashMap<String, String>()
                mapOfHeaders["X-Authorization"] = jwttoken
                val result = FlyAway.delete(context, API, mapOfHeaders, "PushManager.disablePush", null)
                resultObject.setResult(result)
            } catch (e: FlybitsException) {
                Logger.exception("PushManager.disablePush", e)
                resultObject.setFailed(e)
            }
        }
        return resultObject
    }
}
