package com.flybits.android.push.models;

/**
 * The {@code Category} indicates which type of Entity the push notification effects. Below the
 * the complete list of categories and their entities can be found.
 * <ul>
 *     <li>{@link PushCategory#CONTEXT}</li>
 *     <li>{@link PushCategory#KERNEL}</li>
 *     <li>{@link PushCategory#CUSTOM}</li>
 * </ul>
 */
public enum PushCategory {

    /**
     * Indicates that the entity associated to this notification belongs to the {@code CONTEXT}
     * category.
     */
    CONTEXT("context"),

    /**
     * Indicates that the entity associated to this notification belongs to the {@code CUSTOM}
     * category.
     */
    CUSTOM("custom"),

    /**
     * Indicates that the entity associated to this notification belongs to the {@code KERNEL}
     * category.
     */
    KERNEL("kernel"),

    /**
     * Indicates that a unknown category was received with the specified entity.
     */
    UNKNOWN("");

    private final String key;

    /**
     * Constructor that defines the key for each {@code PushCategory} option.
     *
     * @param key the String value representing each {@code PushCategory} option.
     */
    PushCategory(String key) {
        this.key = key;
    }

    /**
     * Get the String representation for the {@code PushCategory} option.
     *
     * @return String representation of the {@code PushCategory} option.
     */
    public String getKey() {
        return this.key;
    }

    /**
     * Get the {@code PushCategory} enum value corresponding to an String representation.
     *
     * @param key the String representation of the {@code PushCategory} enum.
     *
     * @return The {@code PushCategory} enum for the String representation.
     */
    public static PushCategory fromKey(String key) {
        if (key == null){
            return UNKNOWN;
        }

        for(PushCategory type : PushCategory.values()) {
            if(type.getKey().equalsIgnoreCase(key)) {
                return type;
            }
        }
        return UNKNOWN;
    }
}