package com.flybits.android.push

import android.content.Context
import android.content.SharedPreferences
import android.os.Handler
import android.os.Looper

import com.flybits.android.push.db.PushDatabase
import com.flybits.commons.library.api.FlybitsScope
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.models.User

import java.util.HashMap
import java.util.concurrent.Executors

import android.content.Context.MODE_PRIVATE

class PushScope
/**
 * This constructor should be used in the event that the application subscribed to Firebase
 * specific push notifications.
 */
internal constructor() : FlybitsScope("PushScope") {

    private var properties: HashMap<String, String>? = null
    private var autoConnectToPush: Boolean = true

    constructor(properties: HashMap<String, String>) : this() {
        this.properties = properties
    }

    constructor(autoConnect: Boolean) : this() {
        autoConnectToPush = autoConnect
    }

    override fun onConnected(context: Context, user: User) {
        if (autoConnectToPush && user.isOptedIn) {
            PushManager.enableFCMPush(context, properties, null)
        }
    }

    override fun onDisconnected(context: Context, jwtToken: String) {
        disableThenClearData(context, jwtToken, Handler(Looper.getMainLooper()))
    }

    override fun onAccountDestroyed(context: Context, jwtToken: String) {
        disableThenClearData(context, jwtToken, Handler(Looper.getMainLooper()))
    }

    override fun onOptedStateChange(context: Context, optedState: Boolean) {
        if (!optedState) {
            clearDataAsync(context)
        } else {
            PushManager.enableFCMPush(context, properties, null)
        }
    }

    /**
     * Disable push if autoConnectToPush is true, then clear all databases.
     *
     * @param context Context associated to the appliation.
     * @param jwt JWT token associated with current session.
     * @param handler Handler that the callback results will be posted on.
     */
    internal fun disableThenClearData(context: Context, jwt: String, handler: Handler) {
        if (autoConnectToPush) {
            PushManager.disablePush(context, jwt, object : BasicResultCallback {
                override fun onSuccess() {
                    clearDataAsync(context)
                }

                override fun onException(exception: FlybitsException) {}
            }, handler)
        }
    }

    /**
     * Clear the kernel databases on the worker thread.
     *
     * @param context Context associated with the application.
     */
    internal fun clearDataAsync(context: Context) {
        Executors.newSingleThreadExecutor().execute { clearData(context) }
    }

    /**
     * Clear the kernel databases. Warning, this method will run on the caller thread.
     * See [PushScope.clearDataAsync] for asynchronous alternative.
     *
     * @param context Context associated with the application.
     */
    internal fun clearData(context: Context) {
        val preferences = getPushPreferences(context).edit()
        preferences.clear()
        preferences.apply()
        PushDatabase.getDatabase(context).pushDao().clear()
    }

    companion object {

        const val PUSH_PREF = "FLYBITS_PREF_PUSH"
        const val ROOT = "/push"
        @JvmStatic
        val SCOPE = PushScope()

        /**
         * Get the default shared preferences for the application.
         *
         * @param context The context of the application.
         * @return The default SharedPreferences object for the application.
         */
        @JvmStatic
        fun getPushPreferences(context: Context): SharedPreferences {
            return context.getSharedPreferences(PUSH_PREF, MODE_PRIVATE)
        }
    }
}
