package com.flybits.android.push.models;

/**
 * Indicates the action associated to the Push/MQTT notification.
 */
public enum PushAction {

    /**
     * Indicates that a new entity was added. For example, new content is available to the user due
     * to a {@code Rule} status change.
     */
    ADDED("created"),

    /**
     * Indicates that a Push/MQTT notification was delivered with a custom action.
     */
    CUSTOM("custom"),

    /**
     * Indicates that an entity was removed. For example, content that use to be available to the
     * user should no longer be available due to a change in context associated to a {@code Rule}.
     */
    REMOVED("deleted"),

    /**
     * Indicates that a {@code Rule} has been updated.
     */
    RULE_UPDATED("ruleUpdated"),

    /**
     * Indicates that the state of a {@code Rule} based on changing contextual information has been
     * changed from true to false or vice-versa.
     */
    STATUS_CHANGED("statusChanged"),

    /**
     * Indicates that a unknown action was received with the specified entity.
     */
    UNKNOWN(""),

    /**
     * Indicates that an entity has been updated. For example if information associated to content
     * that is currently available to a user has been changed.
     */
    UPDATED("modified");

    private final String key;

    /**
     * Constructor that defines the key for each {@code PushAction} option.
     *
     * @param key the String value representing each {@code PushAction} option.
     */
    PushAction(String key) {
        this.key = key;
    }

    /**
     * Get the String representation for the {@code PushAction} option.
     *
     * @return String representation of the {@code PushAction} option.
     */
    public String getKey() {
        return this.key;
    }

    /**
     * Get the {@code PushAction} enum value corresponding to an String representation.
     *
     * @param key the String representation of the {@code PushAction} enum.
     *
     * @return The {@code PushAction} enum for the String representation.
     */
    public static PushAction fromKey(String key) {
        if (key == null){
            return UNKNOWN;
        }

        for(PushAction type : PushAction.values()) {
            if(type.getKey().equalsIgnoreCase(key)) {
                return type;
            }
        }
        return UNKNOWN;
    }
}
