package com.flybits.android.push.api;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.android.push.deserializations.DeserializePushPreferences;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.flybits.android.push.PushScope.ROOT;

/**
 * The {@code PushPreferenceManager} is responsible actions that can be associated to the
 * retrieval of {@link Push} preferences that indicate any additional information associated to the
 * {@code PushToken} that is sent to the Server for processing whenever the {@code User} logs in.
 */
public class PushPreferenceManager {

    static final     String API                     = ROOT + "/preferences";

    /**
     * Clear the {@code PushPreferences} list which means that the connected user will receive all
     * push notification and no push notifications will be filtered out.
     *
     * @param mContext The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static BasicResult clear(@NonNull final Context mContext,
                                    @NonNull final BasicResultCallback callback){

        final Handler handler   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result result = FlyAway.delete(mContext, API, "PushPreferenceManager.clear", null);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setResult(result);
                        }
                    });

                }catch(final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setFailed(e);
                        }
                    });
                }
            }
        });
        return query;
    }

    /**
     * Retrieve all the {@code PushPreferences} that are stored for the connected user. It is
     * important to make sure that the user is in fact connected to Flybits. This can be achieved
     * through the {@code FlybitsManager#connect()} function.
     *
     * @param mContext The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static ObjectResult<String[]> get(@NonNull final Context mContext,
                                             @NonNull final ObjectResultCallback<String[]> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<String[]> query    = new ObjectResult<String[]>(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result<String[]> result  = FlyAway.get(mContext, API, new DeserializePushPreferences(), "PushPreferenceManager.get", String[].class);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setResult(result);
                        }
                    });
                } catch (final FlybitsException e) {
                     handler.post(new Runnable() {
                        @Override
                        public void run() {
                                query.setFailed(e);
                            }
                        });
                }
            }
        });
        return query;
    }


    /**
     * The function will save a list of "keyword" that should be used to filter push notifications
     * for. For example, if the user does not want to receive notification about "restaurants", then
     * "restaurants" should be including in the {@code listOfOptions} parameter. If this is done the
     * connected user will not receive any push notification that have been labeled with the
     * "restaurants" label.
     *
     * @param mContext The state of the application.
     * @param keywords The list of options that should be used to filter push notification for.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static BasicResult post(@NonNull final Context mContext, @NonNull final ArrayList<String> keywords,
                                      final BasicResultCallback callback) {

        final Handler handler   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    String[] listOfTopics = keywords.toArray(new String[keywords.size()]);
                    final String json = new DeserializePushPreferences().toJson(listOfTopics);
                    final Result result  = FlyAway.post(mContext, API, json, new DeserializePushPreferences(), "PushPreferenceManager.post", String[].class);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setResult(result);
                        }
                    });
                } catch (final FlybitsException e) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });
                }
            }
        });
        return query;
    }


}
