package com.flybits.android.push.utils;

import android.support.annotation.NonNull;

import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link PushQueryParameters} class is used to define all the parameters set for the GET
 * request associated to {@link com.flybits.android.push.models.Push}.
 */
public class PushQueryParameters extends QueryParameters {

    private long epochStartTime;
    private long epochEndTime;
    private SortOrder order;

    private PushQueryParameters(Builder builder) {
        epochStartTime  = builder.epochStartTime;
        epochEndTime    = builder.epochEndTime;
        order           = builder.sort;
        addPaging(builder.limit, builder.offset);
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();
        if (epochEndTime > -1 && epochStartTime > -1) {

            ArrayList<String> listOfStartTime = new ArrayList<>();
            listOfStartTime.add(String.valueOf(epochStartTime));
            params.put("from", listOfStartTime);

            ArrayList<String> listOfEndTime = new ArrayList<>();
            listOfEndTime.add(String.valueOf(epochEndTime));
            params.put("to", listOfEndTime);
        }

        ArrayList<String> listOfSorts = new ArrayList<>();
        listOfSorts.add(order.getKey());
        params.put("order", listOfSorts);
        return params;
    }

    /**
     * The {@code Builder} class that defines all the possible filter options that can be used when
     * retrieving a list of Push Notifications.
     */
    public static class Builder {

        long limit;
        long offset;
        long epochStartTime;
        long epochEndTime;
        SortOrder sort;

        /**
         * Default Constructor that defines an empty {@code Builder}.
         */
        public Builder(){
            limit           = 10;
            offset          = 0;
            epochEndTime    = -1;
            epochStartTime  = -1;
            sort            = SortOrder.DESCENDING;
        }

        /**
         * Adds a paging mechanism to the request based on the number of returned results wanted and the
         * offset of the next X number of results, where X is a limit.
         *
         * @param limit The maximum number of returned objects.
         * @param offset The offset of where the next X number of response objects will be returned
         *               from where X is the limit.
         *
         * @return A {@code Builder} object where additional filter settings can be set.
         */
        public Builder addPaging(long limit, long offset){
            this.limit  = limit;
            this.offset = offset;
            return this;
        }

        /**
         * Adds a time range for which a list of notifications should be retrieved for.
         *
         * @param epochStartTime The start time in epoch which notifications should be retrieved
         *                       from.
         * @param epochEndTime The end time in epoch which notification should be retrieved to.
         *
         * @return A {@code Builder} object where additional filter settings can be set.
         */
        public Builder addTimeRange(long epochStartTime, long epochEndTime){
            this.epochStartTime = epochStartTime;
            this.epochEndTime   = epochEndTime;
            return this;
        }

        /**
         * Build the {@code PushHistoryOptions} object to be used for constructing an HTTP request.
         *
         * @return {@code PushHistoryOptions} object which can be referenced by the SDK to get
         * specific information about a list of Push Notifications.
         */
        public PushQueryParameters build(){
            return new PushQueryParameters(this);
        }

        /**
         * Sets the sort order of the response from the Server.
         *
         * @return A {@code Builder} object where additional filter settings can be set.
         */
        public Builder setSortOrder(@NonNull SortOrder order){
            if (order == null){
                return this;
            }

            this.sort   = order;
            return this;
        }
    }
}
