package com.flybits.android.push;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlybitsScope;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.User;

import static android.content.Context.MODE_PRIVATE;

public class PushScope extends FlybitsScope{

    public static final String ROOT = "/push";

    private enum PushType{

        FCM,

        GCM
    }

    private Context context;
    private String gcmSenderID;
    private PushType type;
    private SharedPreferences flybitsPreferences;

    /**
     * This constructor should be used in the event that the application subscribed to Firebase
     * specific push notifications.
     *
     * @param context The {@link Context} of the application.
     */
    public PushScope(@NonNull Context context) {
        super("PushScope");
        this.context       = context;
        this.type          = PushType.FCM;
        this.flybitsPreferences = getPushPreferences(context);
    }

    /**
     * This constructor should be used in the event that the application subscribed to GCM specific
     * push notification.
     *
     * @param context The {@link Context} of the application.
     * @param gcmSenderID The Application Sender Identifier obtained from Google that indicates
     *                    which corresponding server key should be used for this application.
     */
    public PushScope(@NonNull Context context, String gcmSenderID) {
        super("PushScope");
        this.context            = context;
        this.type               = PushType.GCM;
        this.gcmSenderID        = gcmSenderID;
        this.flybitsPreferences = getPushPreferences(context);
    }

    @Override
    public void onConnected(User user) {
        switch (type){
            case GCM:
                PushManager.enableGCMPush(context, gcmSenderID, new BasicResultCallback() {
                    @Override
                    public void onSuccess() {

                    }

                    @Override
                    public void onException(FlybitsException exception) {

                    }
                });
                break;
            case FCM:
                PushManager.enableFCMPush(context, new BasicResultCallback() {
                    @Override
                    public void onSuccess() {

                    }

                    @Override
                    public void onException(FlybitsException exception) {

                    }
                });
                break;
        }
    }

    @Override
    public void onDisconnected(String jwtToken) {

        PushManager.disablePush(context, jwtToken, new BasicResultCallback() {
            @Override
            public void onSuccess() {

            }

            @Override
            public void onException(FlybitsException exception) {

            }
        });
        clearPushPreferences();
    }

    @Override
    public void onAccountDestroyed(String jwtToken) {
        onDisconnected(jwtToken);
    }

    /**
     * Get the default shared preferences for the application.
     *
     * @param context The context of the application.
     * @return The default SharedPreferences object for the application.
     */
    public static SharedPreferences getPushPreferences(Context context){
        return context.getSharedPreferences("FLYBITS_PREF_PUSH", MODE_PRIVATE );
    }

    private void clearPushPreferences(){
        SharedPreferences.Editor preferences   = getPushPreferences(context).edit();
        preferences.clear();
        preferences.apply();
    }
}
