package com.flybits.android.push.services;

import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Messenger;
import android.support.v4.content.LocalBroadcastManager;

import com.flybits.android.push.api.FlyPushParsing;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlyJWT;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;

import org.eclipse.paho.android.service.MqttAndroidClient;
import org.eclipse.paho.client.mqttv3.IMqttActionListener;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.IMqttToken;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class FlybitsMQTTService extends Service {

    private static final int QOS_WILL               = 1;
    private static final int QOS                    = 0;
    private static final int KEEP_ALIVE            = 30;

    private static boolean serviceRunning           = false;
    private static MQTTConnection connection        = null;
    private final Messenger clientMessenger         = new Messenger(new ClientHandler());

    private final String _TAG                       = "MQTT_Service";
    public static final String MQTT_EVENT           = "mqtt-event";
    public static final String MQTT_CONNECT         = "mqtt-connect";
    public static final String MQTT_QUIT            = "mqtt-quit";
    public static final String IS_SUBSCRIBE         = "mqtt-event:IS_SUBSCRIBE";
    public static final String IS_CONNECT           = "mqtt-connect:IS_CONNECTED";

    @Override
    public void onCreate(){
        super.onCreate();
        Logger.setTag(_TAG).d("OnCreate");
        connection = new MQTTConnection();
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        Logger.setTag(_TAG).d("onStartCommand: " + serviceRunning);

        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiver, new IntentFilter(MQTT_EVENT));
        LocalBroadcastManager.getInstance(this).registerReceiver(mConnectionReceiver, new IntentFilter(MQTT_CONNECT));
        LocalBroadcastManager.getInstance(this).registerReceiver(mCloseReceiver, new IntentFilter(MQTT_QUIT));

        if (isRunning()) {

            if (connection == null) {
                connection = new MQTTConnection();
            }

            try {
                connection.start();
            }catch(RuntimeException exception){
                Logger.exception("FlybitsMQTTService.onStartCommand", exception);
            }

            connection.startConnection();
            setIsRunning(true);

            return START_STICKY;
        }

        super.onStartCommand(intent, flags, startId);
		/*
		 * Start the MQTT Thread.
		 */
        connection.start();
        connection.startConnection();
        setIsRunning(true);

        return START_STICKY;
    }

    // Our handler for received Intents. This will be called whenever an Intent
    // with an action named "custom-event-name" is broadcasted.
    private BroadcastReceiver mConnectionReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            boolean isConnect = intent.getBooleanExtra(IS_CONNECT, false);
            if (isConnect){
                connection.startConnection();
            }else{
                connection.end();
            }
        }
    };

    // Our handler for received Intents. This will be called whenever an Intent
    // with an action named "custom-event-name" is broadcasted.
    private BroadcastReceiver mCloseReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            Logger.setTag(_TAG).d("Quitting Service");
            stopSelf();
        }
    };

    // Our handler for received Intents. This will be called whenever an Intent
    // with an action named "custom-event-name" is broadcasted.
    private BroadcastReceiver mMessageReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            // Get extra data included in the Intent
            String topic            = intent.getStringExtra(TOPIC);
            ArrayList<String> listOfTopics  = intent.getStringArrayListExtra(TOPICS);

            boolean isSubsribe = intent.getBooleanExtra(IS_SUBSCRIBE, false);
            Logger.setTag(_TAG).d("Topic: " + topic);

            if (isSubsribe){

                if (listOfTopics == null || listOfTopics.size() == 0) {
                    connection.subscribeToChannel(topic);
                }else{
                    connection.subscribeToChannel(listOfTopics);
                }
            }else{

                if (listOfTopics == null || listOfTopics.size() == 0) {
                    connection.unsubscribeToChannel(topic);
                }else{
                    connection.unsubscribeToChannel(listOfTopics);
                }
            }
        }
    };

    public MQTTConnection getConnection(){
        return connection;
    }

    @Override
    public void onDestroy() {
        serviceRunning = false;
        Logger.setTag(_TAG).d("onDestroy()");
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiver);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mConnectionReceiver);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mCloseReceiver);
        connection.end();
    }

    @Override
    public IBinder onBind(Intent intent) {
		/*
		 * Return a reference to our client handler.
		 */
        return clientMessenger.getBinder();
    }

    public synchronized static void setIsRunning(boolean isRunning){
        serviceRunning = isRunning;
    }

    public synchronized static boolean isRunning()
    {
        return serviceRunning;
    }


    /*
     * These are the supported messages from bound clients
     */
    public static final int UNSUBSCRIBE     = 4;
    public static final int DISCONNECT      = 3;
    public static final int PUBLISH         = 2;
    public static final int SUBSCRIBE       = 1;
    public static final int CONNECT         = 0;

    /*
     * Fixed strings for the supported messages.
     */
    public static final String TOPIC                = "topic";
    public static final String TOPICS               = "topics";
    public static final String MESSAGE              = "message";
    public static final String DEVICE_ID            = "deviceID";
    public static final String USER_ID              = "userID";

    /*
     * This class handles messages sent to the service by
     * bound clients.
     */
    static class ClientHandler extends Handler {

        @Override
        public void handleMessage(Message msg) {

            switch (msg.what) {

                case SUBSCRIBE:
                case PUBLISH:
                case CONNECT:
                case UNSUBSCRIBE:
                case DISCONNECT:
           		 	/*
           		 	 * These two requests should be handled by
           		 	 * the connection thread, call makeRequest
           		 	 */
                    connection.makeRequest(msg);
                    break;
                //msgHandler.sendMessage(Message.obtain(null, CONNECT));
            }
        }
    }

    enum CONNECT_STATE {
        DISCONNECTED,
        CONNECTING,
        CONNECTED
    }

    protected class MQTTConnection extends Thread {
        private CONNECT_STATE connState;
        private MsgHandler msgHandler = null;
        private MqttAndroidClient client = null;

        private final int MINTIMEOUT = 2000;
        private final int MAXTIMEOUT = 256000;
        private int timeout = MINTIMEOUT;
        private Vector<String> topics = new Vector<>();

        public MQTTConnection(){
            msgHandler = new MsgHandler();
            connState = CONNECT_STATE.DISCONNECTED;
        }

        class MsgHandler extends Handler {

            @Override
            public void handleMessage(final Message msg) {
                switch (msg.what) {
                    case CONNECT:
                        new Thread(new Runnable() {
                            public void run() {
                                Logger.setTag(_TAG).d("Handler - CONNECT(): " + (Looper.myLooper() == Looper.getMainLooper()));
                                connect();
                            }
                        }).start();
                        break;
                    case DISCONNECT:
                        new Thread(new Runnable() {
                            public void run() {
                                Logger.setTag(_TAG).d("Handler - DISCONNECT(): " + (Looper.myLooper() == Looper.getMainLooper()));
                                unconnect();
                            }
                        }).start();
                        break;
                    case SUBSCRIBE:
                        if (msg.obj instanceof ArrayList) {
                            final ArrayList<String> topics = (ArrayList) msg.obj;
                            new Thread(new Runnable() {
                                public void run() {
                                    Logger.setTag(_TAG).d("Handler - SUBSCRIBE(): " + (Looper.myLooper() == Looper.getMainLooper()));
                                    subscribe(topics);
                                }
                            }).start();
                        }
                        break;
                    case UNSUBSCRIBE:
                        if (msg.obj instanceof ArrayList) {
                            final ArrayList<String> topics = (ArrayList) msg.obj;
                            new Thread(new Runnable() {
                                public void run() {
                                    Logger.setTag(_TAG).d("Handler - UNSUBSCRIBE(): " + (Looper.myLooper() == Looper.getMainLooper()));
                                    unsubscribe(topics);
                                }
                            }).start();
                        }
                        break;
                    case PUBLISH:
                        new Thread(new Runnable() {
                            public void run() {
                                Logger.setTag(_TAG).d("Handler - PUBLISH(): " + (Looper.myLooper() == Looper.getMainLooper()));
                            }
                        }).start();
                        break;
                }
            }

            private void connect(){

                Logger.setTag(_TAG).d( "Current Time Out Is: " + timeout);
                if (connState != CONNECT_STATE.CONNECTED) {

                    final String projectID    = SharedElements.getProjectID(getBaseContext());
                    final String deviceID     = SharedElements.getDeviceID(getBaseContext());
                    final String userID       = SharedElements.getUserID(getBaseContext());
                    final String clientID     = deviceID+"_"+projectID;
                    final String will         = userID+"_"+deviceID+"_"+projectID;

                    try {
                        String jwt      = SharedElements.getSavedJWTToken(getBaseContext());
                        if (jwt == null){
                            jwt = "";
                        }

                        Logger.setTag(_TAG).d("Username: " + clientID);
                        Logger.setTag(_TAG).d("Password: " + jwt);

                        String MQTT_url = FlybitsAPIConstants.getMQTTURL();
                        Logger.setTag(_TAG).d("MQTT URL: " + MQTT_url);
                        client = new MqttAndroidClient(getApplicationContext(), MQTT_url, clientID);
                        client.setCallback(new MqttCallback() {
                            @Override
                            public void connectionLost(Throwable cause) {
                                Logger.setTag(_TAG).i("connectionLost: " + cause.getMessage());
                                connState = CONNECT_STATE.DISCONNECTED;
                                sendMessageDelayed(Message.obtain(null, CONNECT), timeout);
                            }

                            @Override
                            public void messageArrived(final String topic, final MqttMessage message) throws Exception {
                                new Thread(new Runnable() {
                                    public void run() {
                                        Logger.setTag(_TAG).i(topic + ":" + message.toString());
                                        String pushNotification = message.toString();

                                        try {
                                            FlyPushParsing.parseMQTTNotification(getApplicationContext(), pushNotification);
                                        }catch(final IllegalArgumentException e){
                                            Logger.exception("FlybitsMQTTService.connect.messageArrived", e);
                                        }
                                    }
                                }).start();
                            }

                            @Override
                            public void deliveryComplete(IMqttDeliveryToken token) {}
                        });

                        MqttConnectOptions mqttConnectOptions = new MqttConnectOptions();
                        mqttConnectOptions.setAutomaticReconnect(true);
                        mqttConnectOptions.setCleanSession(false);
                        mqttConnectOptions.setUserName(clientID);
                        mqttConnectOptions.setPassword(jwt.toCharArray());
                        mqttConnectOptions.setKeepAliveInterval(KEEP_ALIVE);

                        try {
                            mqttConnectOptions.setWill("flybits.mqtt.offline", will.getBytes("UTF-8"), QOS_WILL, false);
                        }catch (UnsupportedEncodingException | IllegalStateException e){
                            Logger.exception("FlybitsMQTTService.connect.UnsupportedEncodingException", e);
                            errorConnecting();
                            return;
                        }

                        client.connect(mqttConnectOptions, null, new IMqttActionListener() {
                            @Override
                            public void onSuccess(IMqttToken asyncActionToken) {
                                Logger.setTag(_TAG).d("Connection: OnSuccess");
                                connState = CONNECT_STATE.CONNECTED;
                                timeout = MINTIMEOUT;

                                /*
                                 * Re-subscribe to previously subscribed topics
                                */
                                if (client.isConnected()) {
                                    Iterator<String> i = topics.iterator();
                                    while (i.hasNext()) {
                                        subscribe(i.next());
                                    }

                                    /*
                                     * Subscribe to user's unique channel
                                     */
                                    if (!topics.contains(clientID)) {
                                        subscribe(clientID);
                                    }
                                }
                            }

                            @Override
                            public void onFailure(IMqttToken asyncActionToken, Throwable exception) {
                                Logger.setTag(_TAG).d("Connection: OnFailed: " + exception.getLocalizedMessage());
                                if (exception instanceof MqttException) {
                                    if (((MqttException) exception).getReasonCode() == MqttException.REASON_CODE_FAILED_AUTHENTICATION) {

                                        final ExecutorService executorService = Executors.newSingleThreadExecutor();
                                        executorService.execute(new Runnable() {
                                            public void run() {
                                                try {
                                                    FlyJWT.refreshJWT(getBaseContext());
                                                } catch (FlybitsException e) {
                                                    Logger.exception("FlybitsMQTTService.connect.onFailure", e);

                                                    if (e instanceof FlybitsDisabledException){
                                                        if (!executorService.isShutdown()){
                                                            stopSelf();
                                                        }
                                                    }
                                                }
                                            }
                                        });
                                    }
                                }
                                errorConnecting();
                            }
                        });
                    } catch (MqttException e) {
                        Logger.exception("FlybitsMQTTService.connect.MqttException", e);
                        errorConnecting();
                        return;
                    }
                }
            }

            private void errorConnecting(){
                if (timeout < MAXTIMEOUT) {
                    timeout *= 2;
                    this.sendMessageDelayed(Message.obtain(null, CONNECT), timeout);
                }else{
                    stopSelf();
                }
            }

            private void unconnect(){

                Logger.setTag(_TAG).d("unconnect()");
                if (client == null){
                    return;
                }

                try {
                    client.setCallback(null);
                    if (client.isConnected()) {
                        try {
                            client.disconnect();
                            client.close();
                        } catch (MqttException e) {
                            Logger.exception("FlybitsMQTTService.unconnect.mqttexception", e);
                        }
                    }
                    Logger.setTag(_TAG).d("Service: STOP");
                }catch(NullPointerException exception){
                    Logger.exception("FlybitsMQTTService.unconnect", exception);
                }
            }

            private void subscribe(ArrayList<String> topicToSubscribeTo){
                Logger.setTag(_TAG).d("subscribeToChannel(): " + topicToSubscribeTo);
                Logger.setTag(_TAG).d("subscribeToChannel() : " + (Looper.myLooper() == Looper.getMainLooper()));

                ArrayList<String> topicsFromBundle = new ArrayList<>(topicToSubscribeTo);

                if (topicsFromBundle.size() > 0){

                    for (String topic : topicsFromBundle){

                        if (!topic.isEmpty()){

                            boolean status = subscribe(topic);
                            if (status)
                            {
                                topics.add(topic);
                            }
                        }
                    }
                }
            }

            private boolean subscribe(String topic) {
                if (client == null || !client.isConnected()){
                    return false;
                }

                try {
                    client.subscribe(topic, QOS);
                } catch (MqttException  e) {
                    Logger.exception("FlybitsMQTTService.subscribe", e);
                    return false;
                }
                return true;
            }

            private void unsubscribe(ArrayList<String> topicToSubscribeTo){
                Logger.setTag(_TAG).d("unsubscribeToChannel(): " + topicToSubscribeTo);

                ArrayList<String> topicsFromBundle = new ArrayList<>(topicToSubscribeTo);

                if (topicsFromBundle.size() > 0){

                    for (String topic : topicsFromBundle){

                        if (!topic.isEmpty()){
                            if (unsubscribe(topic))
                            {
                                topics.add(topic);
                            }
                        }
                    }
                }
            }

            private boolean unsubscribe(String topic) {
                if (client == null || !client.isConnected()){
                    return false;
                }

                try {
                    client.unsubscribe(topic);
                } catch (MqttException e) {
                    Logger.exception("FlybitsMQTTService.unsubscribe", e);
                    return false;
                }
                return true;
            }

            private void publishToChannel(String topicToSendTo, String messageToSend){
                if (topicToSendTo != null) {
                    String topic = topicToSendTo.trim();
                    if (!topic.isEmpty()) {
                        if (messageToSend != null) {
                            String message = messageToSend.trim();
                            if (!message.isEmpty()) {
                                publish(topic, message);
                            }
                        }
                    }
                }
            }

            private boolean publish(String topic, String msg) {

                if (client == null || !client.isConnected()){
                    return false;
                }

                try {
                    MqttMessage message = new MqttMessage();
                    message.setPayload(msg.getBytes());
                    client.publish(topic, message);
                } catch (MqttException e) {
                    Logger.exception("FlybitsMQTTService.publish", e);
                    return false;
                }
                return true;
            }
        }

        public void end()
        {
            Logger.setTag(_TAG).i("end(): " + (Looper.myLooper() == Looper.getMainLooper()));
            msgHandler.removeCallbacksAndMessages(null);
        }

        public void startConnection()
        {
            Logger.setTag(_TAG).i("startConnection(): " + (Looper.myLooper() == Looper.getMainLooper()));
            msgHandler.sendMessage(Message.obtain(null, CONNECT));
        }

        public void subscribeToChannel(String topicSubscribed)
        {
            ArrayList<String> listOfTopics  = new ArrayList<>();
            listOfTopics.add(topicSubscribed);
            subscribeToChannel(listOfTopics);
        }

        public void subscribeToChannel(ArrayList<String> topics)
        {

            Message msg = Message.obtain(null, SUBSCRIBE);
            msg.obj     = topics;
            msgHandler.sendMessage(msg);
        }

        public void unsubscribeToChannel(String topicUnsubscribed)
        {
            ArrayList<String> listOfTopics  = new ArrayList<>();
            listOfTopics.add(topicUnsubscribed);
            unsubscribeToChannel(listOfTopics);
        }

        public void unsubscribeToChannel(ArrayList<String> topics)
        {
            Message msg = Message.obtain(null, UNSUBSCRIBE);
            msg.obj     = topics;
            msgHandler.sendMessage(msg);
        }

        public void makeRequest(Message msg)
        {
			/*
			 * It is expected that the caller only invokes
			 * this method with valid msg.what.
			 */
            msgHandler.sendMessage(Message.obtain(msg));
        }
    }


}
