package com.flybits.android.push.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.android.push.PushScope;
import com.flybits.android.push.deserializations.DeserializePushToken;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.Result;

import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class is used to define a GCM push token which should be transferred from the device, once
 * it is obtained from Google, to the Flybits Push Server. Through this {@code PushToken} class the
 * Flybits Push Server will be able to know to which {@code User} to send Push notification to
 * based on specific Server and Business logic.
 *
 * @since 2.1.8
 */
public class PushToken implements Parcelable {

    public static final String  API                = PushScope.ROOT + "/token";

    /**
     * The properties that are associated to this device token.
     */
    private HashMap<String, String> data;

    /**
     * The GCM/APNS {@code PushToken}.
     */
    private String deviceToken;

    /**
     * Indicates through which service the {@code PushToken} was generated from.
     */
    private PushType network;

    /**
     * Constructor used to define the token obtained from Google's GCM service.
     *
     * @param token The GCM token obtained from Google.
     */
    public PushToken(@NonNull String token){
        deviceToken = token;
        network = PushType.GCM;
    }

    /**
     * Get additional data that is associated to the {@code PushToken}. This data is added by an
     * application as additional metadata.
     *
     * @return The {@code HashMap} containing additional information associated to the
     * {@code PushToken}.
     */
    public HashMap<String, String> getData() {
        return data;
    }

    /**
     * Get the GCM/FCM token associated to the device which is retrieved from a Google server.
     *
     * @return The GCM/FCM token associated to this device.
     */
    public String getDeviceToken() {
        return deviceToken;
    }

    /**
     * Get the type of token that is associated to the device.
     *
     * @return The {@link PushType} of the saved token of the device.
     */
    public PushType getType() {
        return network;
    }

    /**
     * Constructor used to define the token obtained from Google's GCM service.
     *
     * @param token The GCM token obtained from Google.
     * @param data Key-Value pairs that is associated to this specific instance of Push
     *             Notifications. These Key-Value pair allow clients to define specific
     *             characteristics associated to their Flybits instance. In many cases this is
     *             useful for clients to require the Flybits Push system to notify the client's own
     *             Push Notification system from where their notification will initiate.
     */
    public PushToken(String token, HashMap<String, String> data){
        deviceToken = token;
        network     = PushType.GCM;
        this.data   = data;
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    protected PushToken(Parcel in){
        deviceToken = in.readString();
        network     = PushType.fromKey(in.readInt());
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }


    /**
     * Flatten this {@code PushToken} into a Parcel.
     *
     * @param out The Parcel in which the {@code PushToken} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(deviceToken);
        out.writeInt(network.getKey());
    }

    /**
     * Parcelable.Creator that instantiates {@code PushToken} objects
     */
    public static final Parcelable.Creator<PushToken> CREATOR = new Parcelable.Creator<PushToken>() {
        public PushToken createFromParcel(Parcel in) {
            return new PushToken(in);
        }

        public PushToken[] newArray(int size) {
            return new PushToken[size];
        }
    };

    /**
     *
     * @param mContext
     * @param callback
     * @return
     */
    public static ObjectResult<PushToken> get(@NonNull final Context mContext, final ObjectResultCallback<PushToken> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<PushToken> query = new ObjectResult<PushToken>(mContext, callback, executorService);
        query.setService(executorService);

        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result<PushToken> result = FlyAway.get(mContext, API, new DeserializePushToken(), "PushToken.get", PushToken.class);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setResult(result);
                        }
                    });
                } catch (final FlybitsException e) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setFailed(e);
                        }
                    });
                }
            }
        });
        return query;
    }
}
