package com.flybits.android.push;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.AsyncTask;
import android.support.annotation.NonNull;

import com.flybits.android.push.db.PushDatabase;
import com.flybits.commons.library.api.FlybitsScope;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.User;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.util.ArrayList;
import java.util.List;

import static android.content.Context.MODE_PRIVATE;

public class PushScope extends FlybitsScope{

    public static final String ROOT = "/push";

    private enum PushType{

        FCM,

        GCM
    }

    public static PushScope SCOPE = new PushScope();

    private String gcmSenderID;
    private PushType type;

    /**
     * This constructor should be used in the event that the application subscribed to Firebase
     * specific push notifications.
     */
    public PushScope() {
        super("PushScope");
        this.type          = PushType.FCM;
    }

    /**
     * This constructor should be used in the event that the application subscribed to GCM specific
     * push notification.
     *
     * @param gcmSenderID The Application Sender Identifier obtained from Google that indicates
     *                    which corresponding server key should be used for this application.
     */
    public PushScope(String gcmSenderID) {
        super("PushScope");
        this.type               = PushType.GCM;
        this.gcmSenderID        = gcmSenderID;
    }

    @Override
    public void onConnected(Context context, User user) {
        switch (type){
            case GCM:
                PushManager.enableGCMPush(context, gcmSenderID, new BasicResultCallback() {
                    @Override
                    public void onSuccess() {

                    }

                    @Override
                    public void onException(FlybitsException exception) {

                    }
                });
                break;
            case FCM:
                PushManager.enableFCMPush(context, new BasicResultCallback() {
                    @Override
                    public void onSuccess() {

                    }

                    @Override
                    public void onException(FlybitsException exception) {

                    }
                });
                break;
        }
    }

    @Override
    public void onDisconnected(final Context context, String jwtToken) {

        PushManager.disablePush(context, jwtToken, new BasicResultCallback() {
            @Override
            public void onSuccess() {
                clearPushPreferences(context);
            }

            @Override
            public void onException(FlybitsException exception) {

            }
        });
    }

    @Override
    public void onAccountDestroyed(Context context, String jwtToken) {
        onDisconnected(context, jwtToken);
    }

    /**
     * Get the default shared preferences for the application.
     *
     * @param context The context of the application.
     * @return The default SharedPreferences object for the application.
     */
    public static SharedPreferences getPushPreferences(Context context){
        return context.getSharedPreferences("FLYBITS_PREF_PUSH", MODE_PRIVATE );
    }

    private void clearPushPreferences(final Context context){
        SharedPreferences.Editor preferences   = getPushPreferences(context).edit();
        preferences.clear();
        preferences.apply();
        new ClearDatabase(context).execute();
    }

    private class ClearDatabase extends AsyncTask<Void, Void, Void> {

        private Context context;

        public ClearDatabase(Context context){
            this.context    = context;
        }

        protected Void doInBackground(Void... urls) {
            PushDatabase.getDatabase(context).pushDao().clear();
            return null;
        }
    }
}
