package com.flybits.android.push.utils;

import android.support.annotation.NonNull;

import com.flybits.android.push.db.caching.PushCacheLoader;
import com.flybits.commons.library.models.internal.QueryBuilder;
import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link PushQueryParameters} class is used to define all the parameters set for the GET
 * request associated to {@link com.flybits.android.push.models.Push}.
 */
public class PushQueryParameters extends QueryParameters {

    private long epochStartTime;
    private long epochEndTime;
    private SortOrder order;

    private PushQueryParameters(Builder builder) {
        super(builder);
        epochStartTime  = builder.epochStartTime;
        epochEndTime    = builder.epochEndTime;
        order           = builder.sort;
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){
        Map<String, ArrayList<String>> params = super.getQueryParams();
        if (epochEndTime > -1 && epochStartTime > -1) {

            ArrayList<String> listOfStartTime = new ArrayList<>();
            listOfStartTime.add(String.valueOf(epochStartTime));
            params.put("from", listOfStartTime);

            ArrayList<String> listOfEndTime = new ArrayList<>();
            listOfEndTime.add(String.valueOf(epochEndTime));
            params.put("to", listOfEndTime);
        }

        ArrayList<String> listOfSorts = new ArrayList<>();
        listOfSorts.add(order.getKey());
        params.put("order", listOfSorts);
        return params;
    }

    /**
     * The {@code Builder} class that defines all the possible filter options that can be used when
     * retrieving a list of Push Notifications.
     */
    public static class Builder extends QueryBuilder<Builder> {

        long epochStartTime;
        long epochEndTime;
        SortOrder sort;

        /**
         * Default Constructor that defines an empty {@code Builder}.
         */
        public Builder(){
            super();
            epochEndTime    = -1;
            epochStartTime  = -1;
            sort            = SortOrder.DESCENDING;
        }

        /**
         * Adds a time range for which a list of notifications should be retrieved for.
         *
         * @param epochStartTime The start time in epoch which notifications should be retrieved
         *                       from.
         * @param epochEndTime The end time in epoch which notification should be retrieved to.
         *
         * @return A {@code Builder} object where additional filter settings can be set.
         */
        public Builder addTimeRange(long epochStartTime, long epochEndTime){
            this.epochStartTime = epochStartTime;
            this.epochEndTime   = epochEndTime;
            return this;
        }

        /**
         * Build the {@code PushHistoryOptions} object to be used for constructing an HTTP request.
         *
         * @return {@code PushHistoryOptions} object which can be referenced by the SDK to get
         * specific information about a list of Push Notifications.
         */
        public PushQueryParameters build(){
            return new PushQueryParameters(this);
        }

        /**
         * Sets the sort order of the response from the Server.
         *
         * @return A {@code Builder} object where additional filter settings can be set.
         */
        public Builder setSortOrder(@NonNull SortOrder order){
            if (order == null){
                return this;
            }

            this.sort   = order;
            return this;
        }

        public Builder setCachingKey(){
            super.setCachingKey(PushCacheLoader.PUSH_CACHE_KEY);
            return this;
        }
    }
}
