package com.flybits.android.push.models;

import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.android.push.PushScope;
import com.flybits.android.push.db.PushDatabase;
import com.flybits.android.push.db.caching.PushCacheLoader;
import com.flybits.android.push.deserializations.DeserializePushNotification;
import com.flybits.android.push.deserializations.DeserializePushNotificationCustomFields;
import com.flybits.android.push.models.results.PushResult;
import com.flybits.android.push.utils.PushQueryParameters;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class is used to represent a push notification obtained from the Flybits Push Server through
 * either GCM or MQTT depending whether the notification was sent as either a foreground or
 * background notification.
 *
 * <p>A push notification can either be represented through a {@link android.app.Notification} found
 * within the OS' notification tray.
 */
//TODO: Add Comments to Getters and Setters.
@Entity(tableName = "push")
public class Push implements Parcelable {

    static final     String API                   = PushScope.ROOT + "/notifications";

    private PushAction action;
    private HashMap<String, String> customFields;
    private String customFieldsAsString;
    private PushCategory category;
    private PushEntity entity;
    @PrimaryKey
    @NonNull
    private String id;
    private String message;
    private long timestamp;
    private String title;
    private long version;

    /**
     * Default constructor used to allow ROOM to auto-initiate a {@code Push} object from Cache.
     */
    public Push(){}

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param title The title of an alert notification.
     * @param alert The message of an alert notification.
     */
    @Ignore
    public Push(String entity, String action, String category, long version, long timestamp, String title, String alert) {
        this.entity             = PushEntity.fromKey(entity);
        this.action             = PushAction.fromKey(action);
        this.category           = PushCategory.fromKey(category);
        this.version            = version;
        this.timestamp          = timestamp;
        this.title              = title;
        this.message            = alert;
        this.customFields       = new HashMap<>();
    }

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param body The body, represented by a String, of the content of the push notification.
     * @param title The title of an alert notification.
     * @param alert The message of an alert notification.
     */
    @Ignore
    public Push(String entity, String action, String category, long version, long timestamp, String title, String alert, String body) {
        this(entity, action, category, version, timestamp, title, alert);
        this.customFieldsAsString   = body;

        DeserializePushNotificationCustomFields deserializer    = new DeserializePushNotificationCustomFields();
        if ( deserializer.fromJson(this.customFieldsAsString) != null){
            this.customFields   =deserializer.fromJson(this.customFieldsAsString);
        }
    }

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unqiue identifier for this {@code Push} object.
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param title The title of an alert notification.
     * @param alert The message of an alert notification.
     */
    @Ignore
    public Push(String id, String entity, String action, String category, long version, long timestamp, String title, String alert) {
        this(entity, action, category, version, timestamp, title, alert);
        this.id             = id;
    }

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unqiue identifier for this {@code Push} object.
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param title The title of an alert notification.
     * @param alert The message of an alert notification.
     * @param body The body, represented by a String, of the content of the push notification.
     */
    @Ignore
    public Push(String id, String entity, String action, String category, long version, long timestamp, String title, String alert, String body) {
        this(entity, action, category, version, timestamp, title, alert, body);
        this.id                     = id;
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    @Ignore
    public Push(Parcel in) {
        entity                  = PushEntity.fromKey(in.readString());
        action                  = PushAction.fromKey(in.readString());
        category                = PushCategory.fromKey(in.readString());
        version                 = in.readLong();
        timestamp               = in.readLong();
        customFieldsAsString    = in.readString();
        message                 = in.readString();
        title                   = in.readString();
        id                      = in.readString();

        DeserializePushNotificationCustomFields deserializeResult = new DeserializePushNotificationCustomFields();
        customFields = deserializeResult.fromJson(customFieldsAsString);
    }

    /**
     * Get the {@link PushAction} that has occurred on the {@code entity}.
     *
     * @return The {@link PushAction} of the received {@code entity}.
     */
    public PushAction getAction(){
        return action;
    }

    /**
     * Get the body object represented as a String in case the developer would like to perform their
     * own deserializations. This information is set through the custom fields portion of the push
     * notification. It may be null in the event that no custom fields were set.
     *
     * @return The String representation of the {@code Push}'s custom fields.
     */
    public String getCustomFieldsAsString(){
        return customFieldsAsString;
    }

    /**
     * Get the {@link PushCategory} that the {@code entity} belongs to.
     *
     * @return The {@link PushCategory} of the received {@code entity}.
     */
    public PushCategory getCategory(){
        return category;
    }

    /**
     * Get a HashMap of custom fields that have been set within the Experience Studio. The custom
     * fields are represented with as a key-value system.
     *
     * @return The {@code HashMap} that contains Key-Value pairs of the Custom Fields object
     * assigned in the Experience Studio.
     */
    public HashMap<String, String> getCustomFields(){
        return customFields;
    }

    /**
     * Get the {@link PushEntity} that the {@code entity} belongs to.
     *
     * @return The {@link PushEntity} of the received {@code entity}.
     */
    public PushEntity getEntity(){
        return entity;
    }

    /**
     * The unique identifier representing this push notification. This is unique for the push
     * notification across the ecosystem.
     *
     * @return The unique identifier of the {@code Push} Notification. This identifier can be used
     * later on to retrieve/delete content associated to this {@code Push} notification.
     */
    public String getId(){
        return id;
    }

    /**
     * Get the message of an notification. Alert notification should be displayed within the
     * notification bar notification's header field. In silent notifications this field will be
     * null.
     *
     * @return The message of the push notification.
     */
    public String getMessage(){
        return message;
    }

    /**
     * Get the timestamp which indicates when the push notification was generated.
     *
     * @return The epoch time of when the push notification was generated.
     */
    public long getTimestamp(){
        return timestamp;
    }

    /**
     * Get the title of an Push/MQTT notification. The notification should be displayed within the
     * notification bar notification's title field. In silent notifications this field will be
     * null.
     *
     * @return The title of the push notification.
     */
    public String getTitle(){
        return title;
    }

    /**
     * Get the version of the {@code Push} notification. This version is used to identify the push
     * notification structure and can be used to parse information based on this.
     *
     * @return The version number of the Push structure.
     */
    public long getVersion(){
        return version;
    }


    public void setAction(PushAction action) {
        this.action = action;
    }

    public void setCustomFieldsAsString(String customFieldsAsString) {
        this.customFieldsAsString = customFieldsAsString;
    }

    public void setCategory(PushCategory category) {
        this.category = category;
    }

    public void setEntity(PushEntity entity) {
        this.entity = entity;
    }

    public void setId(String id) {
        this.id = id;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public void setVersion(long version) {
        this.version = version;
    }

    public void setCustomFields(HashMap<String, String> customFields) {
        this.customFields = customFields;
    }

    @Override
    public String toString() {
        return  "{ \"version\" : " + version +
                ", \"timestamp\" : " + timestamp +
                ", \"entity\" : \"" + entity.getKey()+"\""+
                ", \"action\" : \"" + action.getKey()+"\""+
                ", \"category\" : \"" + category.getKey()+"\""+
                ", \"title\" : \"" + title+"\""+
                ", \"alert\" : \"" + message+"\""+
                ", \"body\" : " + customFieldsAsString +
                '}';
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    /**
     * Flatten this {@code Push} into a Parcel.
     *
     * @param out The Parcel in which the {@code Push} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(entity.getKey());
        out.writeString(action.getKey());
        out.writeString(category.getKey());
        out.writeLong(version);
        out.writeLong(timestamp);
        out.writeString(customFieldsAsString);
        out.writeString(message);
        out.writeString(title);
        out.writeString(id);
    }

    /**
     * Parcelable.Creator that instantiates {@code Push} objects
     */
    public static final Parcelable.Creator<Push> CREATOR = new Parcelable.Creator<Push>() {
        public Push createFromParcel(Parcel in) {
            return new Push(in);
        }

        public Push[] newArray(int size) {
            return new Push[size];
        }
    };

    /**
     * Retrieve all the {@link com.flybits.android.push.models.Push} notifications that have been
     * sent to the connected user. It is important to make sure that the user is in fact connected
     * to Flybits. This can be achieved through the {@code FlybitsManager#connect()} function.
     *
     * @param context The state of the application.
     * @param params The {@link PushQueryParameters} which allow you to easily define the pagination
     *                schema for the request you are trying to make.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static PushResult get(@NonNull final Context context, @NonNull final PushQueryParameters params,
                                 final PagedResultCallback<Push> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final PushResult query = new PushResult(context, callback, executorService);
        query.setService(executorService);

        executorService.execute(new Runnable() {
            public void run() {

                try {
                    DeserializePushNotification singleDersializer   = new DeserializePushNotification();
                    final Result<PagedResponse<Push>> result = FlyAway.get(context, API, params, new DeserializePagedResponse<Push>(singleDersializer), "Push.get");


                    //Update the DB for caching/observing purposes
                    if (result.getStatus() == RequestStatus.COMPLETED) {
                        if ((params.getQueryParams().get("offset") == null
                                || params.getQueryParams().get("offset").size() == 0
                                || params.getQueryParams().get("offset").get(0).equals("0"))
                                && (params.getCachingKey() != null)) {

                            PushDatabase.getDatabase(context).pushDao().clear();
                            CommonsDatabase.getDatabase(context).cachingEntryDAO().deleteAllByCachingKey(params.getCachingKey());
                        }

                        if (params.getCachingKey() != null) {
                            ArrayList<CachingEntry> entries = new ArrayList<>();
                            for (Push content : result.getResult().getItems()) {
                                entries.add(new CachingEntry(PushCacheLoader.PUSH_CACHE_KEY, content.getId()));
                            }

                            CommonsDatabase.getDatabase(context).cachingEntryDAO().insert(entries);
                            PushDatabase.getDatabase(context).pushDao().insert(result.getResult().getItems());
                        }

                    }
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setResult(result, params);
                        }
                    });
                } catch (final FlybitsException e) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            query.setFailed(e);
                        }
                    });
                }
            }
        });
        return query;
    }

    /**
     * Retrieve all the {@link com.flybits.android.push.models.Push} notifications that have been
     * sent to the connected user. It is important to make sure that the user is in fact connected
     * to Flybits. This can be achieved through the {@code FlybitsManager#connect()} function.
     *
     * @param mContext The state of the application.
     * @param params The {@link PushQueryParameters} which allow you to easily define the pagination
     *                schema for the request you are trying to make.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static PushResult get(@NonNull final Context mContext, @NonNull final PushQueryParameters params) {
       return get(mContext, params, null);
    }

    /**
     * Deletes a specific {@link com.flybits.android.push.models.Push} notification from the list of
     * notifications sent to the connected user. It is important to make sure that the user is in
     * fact connected to Flybits. This can be achieved through the {@code FlybitsManager#connect()}
     * function.
     *
     * @param mContext The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public BasicResult delete(@NonNull final Context mContext, @NonNull final BasicResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult resultObject    = new BasicResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result result = FlyAway.delete(mContext, API, "Push.delete", id);

                    if (result.getStatus() == RequestStatus.COMPLETED){
                        PushDatabase.getDatabase(mContext).pushDao().delete(Push.this);
                    }
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            resultObject.setResult(result);
                        }
                    });
                }catch(final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });

                }
            }
        });
        return resultObject;
    }
}
