package com.flybits.android.push;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.android.push.api.FlyPushParsing;
import com.flybits.android.push.deserializations.DeserializePushToken;
import com.flybits.android.push.exceptions.FlybitsPushException;
import com.flybits.android.push.models.Push;
import com.flybits.android.push.models.PushToken;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.deserializations.IDeserializer;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;
import com.google.android.gms.gcm.GoogleCloudMessaging;
import com.google.android.gms.iid.InstanceID;
import com.google.firebase.iid.FirebaseInstanceId;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.flybits.android.push.PushScope.getPushPreferences;
import static com.flybits.android.push.models.PushToken.API;
import static com.flybits.android.push.services.gcm.FlybitsGCMRegistrationIntentService.SENT_TOKEN_TO_SERVER;

/**
 * The {@code PushManager} is responsible for all Push related mechanisms such as registering for
 * push notifications, parsing push notifications, and communicating with the GCM servers.
 */
public class PushManager {
    static final String PREF_FIREBASE_TOKEN       = "FIREBASE_TOKEN";
    static final String PREF_PUSH_GCM_APP_ID      = "prefApplicationID";
    static final String PREF_PUSH_GCM_TOKEN       = "prefGCMRegistrationID";
    static final String PREF_PUSH_APP_VERSION     = "prefAppVersion";

    /**
     * Disables Flybits-based push notifications including GCM and foreground notifications. This
     * process will remove the GCM token from the Flybits Push service as well as stopping the
     * foreground Push Service that is internal to the SDK.
     */
    public static BasicResult disablePush(final Context mContext, final String jwttoken, final BasicResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult resultObject    = new BasicResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    HashMap<String, String> mapOfHeaders    = new HashMap<String, String>();
                    mapOfHeaders.put("X-Authorization", jwttoken);
                    final Result result = FlyAway.delete(mContext, API, mapOfHeaders, "PushManager.disablePush", null);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            resultObject.setResult(result);
                        }
                    });
                }catch (final FlybitsException e){
                    Logger.exception("PushManager.disablePush", e);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            resultObject.setFailed(e);
                        }
                    });
                }
            }
        });
        return resultObject;
    }

    /**
     * Enable Flybits-based push notifications including GCM and foreground notifications. This
     * process will retrieve the GCM token from Google and pass it to the Flybits Push Service.
     * Additionally, the Flybits foreground Push service is started.
     */
    public static BasicResult enableGCMPush(final Context mContext, final String appID, final BasicResultCallback callback){
        final SharedPreferences.Editor sharedPreferencesEditor = getPushPreferences(mContext).edit();

        try {
            InstanceID instanceID = InstanceID.getInstance(mContext);
            if (instanceID == null){
                return returnError(mContext, callback);
            }

            final String token = instanceID.getToken(appID,
                    GoogleCloudMessaging.INSTANCE_ID_SCOPE,
                    null);
            if (token == null){
                return returnError(mContext, callback);
            }

            Logger.setTag("PushManager").i("GCM Registration Token: " + token);

            final Handler handler = new Handler(Looper.getMainLooper());
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            final BasicResult resultObject = new BasicResult(mContext, callback, executorService);
            executorService.execute(new Runnable() {
                @Override
                public void run() {

                    try {
                        PushToken resultSerializationObj = new PushToken(token, null);
                        DeserializePushToken deserialization = new DeserializePushToken();
                        final String json = deserialization.toJson(resultSerializationObj);
                        final Result pushResult = FlyAway.post(mContext, API, json, null, "PushManager.enableFCM", null);
                        sharedPreferencesEditor.putBoolean(SENT_TOKEN_TO_SERVER, pushResult.getStatus() == RequestStatus.COMPLETED).apply();

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                resultObject.setResult(pushResult);
                            }
                        });

                    } catch (final FlybitsException e) {
                        Logger.exception("PushManager.enableGCMPush", e);
                        sharedPreferencesEditor.putBoolean(SENT_TOKEN_TO_SERVER, false).apply();
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                resultObject.setFailed(e);
                            }
                        });
                    }
                }
            });

            return resultObject;
        }catch (IOException | NullPointerException e){
            return returnError(mContext, callback);
        }
    }

    /**
     * Enable Flybits-based push notifications including GCM and foreground notifications. This
     * process will retrieve the GCM token from Google and pass it to the Flybits Push Service.
     * Additionally, the Flybits foreground Push service is started.
     */
    public static BasicResult enableFCMPush(final Context mContext, BasicResultCallback callback){
        final String firebaseToken  = FirebaseInstanceId.getInstance().getToken();
        final String token          = (firebaseToken == null || firebaseToken.length() == 0) ? PushManager.getSavedFirebaseToken(mContext) : firebaseToken;
        if (token != null && token.length() > 0) {

            final Handler handler = new Handler(Looper.getMainLooper());
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            final BasicResult resultObject = new BasicResult(mContext, callback, executorService);
            executorService.execute(new Runnable() {
                @Override
                public void run() {
                    try {

                        PushToken resultSerializationObj    = new PushToken(token, null);
                        DeserializePushToken deserialization = new DeserializePushToken();
                        final String json = deserialization.toJson(resultSerializationObj);
                        final Result pushResult = FlyAway.post(mContext, API, json, null, "PushManager.enableFCM", null);

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                resultObject.setResult(pushResult);
                            }
                        });
                    }catch (final FlybitsException e){

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                resultObject.setFailed(e);
                            }
                        });
                    }
                }
            });
            return resultObject;
        }
        return returnError(mContext, callback);
    }

    /**
     * Get the application Sender identifier of the GCM registered application.
     *
     * @param context The {@link Context} of the application.
     * @return The GCM application identifier.
     */
    public static String getAppID(Context context){
        final SharedPreferences prefs = getPushPreferences(context);
        return prefs.getString(PREF_PUSH_GCM_APP_ID, FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * This method retrieves the application version. This is an important aspect of GCM token
     * retrieval as token are not suppose to change for the same version of the application.
     * Therefore, this method is used to confirm whether or not a new token should be retrieved.
     *
     * @param context The {@link Context} of the application.
     * @return The application code as an integer.
     * @throws PackageManager.NameNotFoundException when the package cannot be found. This is highly
     * likely as the application only search for its own package name.
     */
    public static int getAppVersion(Context context) throws PackageManager.NameNotFoundException {
        PackageInfo packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
        return packageInfo.versionCode;
    }

    /**
     * Provides a mechanism for figuring out whether or a saved GCM token is still valid. This takes
     * into account whether or not the application version has changed and therefore indicates if
     * the GCM token should be refreshed.
     *
     * @param context The {@link Context} of the application.
     * @return The stored GCM token or if no GCM token is saved then an empty string will be
     * retrieved. Additionally, if the GCM token was saved under a previous application version an
     * empty String will be returned as well as it means that a new Token should be retrieved.
     */
    public static String getRegistrationId(Context context) {
        final SharedPreferences prefs = getPushPreferences(context);
        String registrationId = prefs.getString(PREF_PUSH_GCM_TOKEN, FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING);
        if (registrationId.isEmpty()) {
            return FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING;
        }

        int registeredVersion = prefs.getInt(PREF_PUSH_APP_VERSION, Integer.MIN_VALUE);

        try {
            int currentVersion = getAppVersion(context);
            if (registeredVersion != currentVersion) {
                return FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING;
            }
        }catch(PackageManager.NameNotFoundException e){
            return FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING;
        }

        return registrationId;
    }

    /**
     * Get the GCM token that is registered to the application's device based on the
     * {@link #getAppVersion(Context)} method. Once the application retrieves the GCM Token from
     * Google it should be stored within the application's preferences.
     *
     * @param mContext The {@link Context} of the application.
     * @return The stored GCM token or if no GCM token is saved then an empty string will be
     * retrieved.
     */
    public static String getStoredGCMToken(Context mContext) {
        final SharedPreferences prefs = getPushPreferences(mContext);
        return prefs.getString(PREF_PUSH_GCM_TOKEN, FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * Parse a FCM {@link Push} notification that is received from the Flybits platform. This push
     * notification is provided in a specific Flybits format which the SDK can parse. If the push
     * notification is not a Flybits notification and therefore cannot be parsed a
     * {@link FlybitsPushException} will be thrown within the
     * {@code PendingRequest.RequestCallbackWithResponse#onException(FlybitsException)}.
     *
     * @param mContext The {@link Context} of the application.
     * @param map The {@link Map} which contains the properties sent from the Flybits
     *               platform. This should not be tempered with and should be sent directly to this
     *               method from the FCM receiver.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which
     *                 indicates the reason for failure.
     * @param <T> The {@code body} of the custom fields option. Some push notification may
     *            provide additional fields with the request in order provide mechanisms for
     *            deep-linking hence a unique structure can be provided.
     */
    public static <T extends IDeserializer> void parsePushNotification(@NonNull Context mContext, @NonNull Map map, ObjectResultCallback<Push> callback){
        try {
            Push push = FlyPushParsing.parsePushNotification(mContext, map);
            if (callback != null) {
                callback.onSuccess(push);
            }
        }catch(FlybitsPushException e){
            if (callback != null) {
                callback.onException(e);
            }
        }
    }

    /**
     * Set the application GCM Sender Identifier.
     *
     * @param mContext The {@link Context} of the application.
     * @param appID The GCM Application Sender ID.
     */
    public static void setAppGCMID(Context mContext, String appID){
        final SharedPreferences.Editor prefs = getPushPreferences(mContext).edit();
        prefs.putString(PREF_PUSH_GCM_APP_ID, appID);
        prefs.apply();
    }

    /**
     * Sets the GCM Token within the application's SharedPreferences so that it can be retrieved
     * later when deleting it from the Server.
     *
     * @param context The {@link Context} of the application.
     * @param regId The GCM token to be set.
     */
    public static void setGCMToken(Context context, String regId) {
        final SharedPreferences.Editor prefs = getPushPreferences(context).edit();
        try {
            int appVersion = getAppVersion(context);
            prefs.putString(PREF_PUSH_GCM_TOKEN, regId);
            prefs.putInt(PREF_PUSH_APP_VERSION, appVersion);
            prefs.apply();
        }catch (PackageManager.NameNotFoundException e){
            Logger.exception("PushManager.setGCMToken", e);
        }
    }

    /**
     * Get the previously saved Firebase token. Firebase tokens are obtained by if the
     * AndroidManifest.xml file is configured for Firebase.
     *
     * @return The saved Firebase token or "" if no Firebase token is saved.
     */
    public static String getSavedFirebaseToken(Context mContext){
        SharedPreferences preferences   = getPushPreferences(mContext);
        return preferences.getString(PREF_FIREBASE_TOKEN, FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * Set the Firebase Cloud Messaging (FCM) token for the device.
     *
     * @param mContext The {@link Context} of the application.
     * @param firebaseToken The FCM token.
     */
    public static void setFirebaseToken(Context mContext, @NonNull String firebaseToken) {
        final SharedPreferences.Editor prefs = getPushPreferences(mContext).edit();
        prefs.putString(PREF_FIREBASE_TOKEN, firebaseToken);
        prefs.apply();
    }

     @Nullable static String getPushToken(Context mContext) {

        SharedPreferences preferences   = getPushPreferences(mContext);
        if (!getSavedFirebaseToken(mContext).equals(FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING)){
            return preferences.getString(PREF_FIREBASE_TOKEN, FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING);
        }else if(!getRegistrationId(mContext).equals(FlyingPushConstants.PREF_DEFAULT_EMPTY_STRING)){
            return getRegistrationId(mContext);
        }
        return null;
    }

    private static BasicResult returnError(Context mContext, BasicResultCallback callback){
        BasicResult result = new BasicResult(mContext,callback);
        result.setFailed(new FlybitsException("Could not obtain InstanceID."));
        return result;
    }
}
