package com.flybits.android.kernel

import android.content.Context
import com.flybits.android.kernel.models.Content
import com.flybits.commons.library.analytics.Analytics

/**
 * This class is responsible for tracking Content analytics. Example of this include:
 * content viewed, content engaged, or content component engaged.
 *
 * @param context The context of the application.
 */
class ContentAnalytics(context: Context): Analytics(context) {

    companion object {
        const val analyticsScope = "contentAnalytics"
    }

    private enum class Action(val actionName: String){
        VIEWED("viewed")
        , ENGAGED("engaged")
        , FULFILLED("fulfilled")
        , COMPONENT_ENGAGED("componentEngaged")
        , COMPONENT_FULFILLED("componentFulfilled")
    }

    private fun track(action: Action, contentId: String, timestamp: Long){
        super.track(AnalyticsEvent(analyticsScope, "query.${action.actionName}.$contentId", true, timestamp))
    }

    /**
     * Track a {@link Content} item being viewed using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackViewed(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, content.id, timestamp)
    }

    /**
     * Track a {@link Content} item being viewed using an id.
     *
     * @param contentId The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackViewed(contentId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, contentId, timestamp)
    }

    /**
     * Track a {@link Content} item being engaged using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackEngaged(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, content.id, timestamp)
    }

    /**
     * Track a {@link Content} item being engaged using an id.
     *
     * @param contentId The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackEngaged(contentId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, contentId, timestamp)
    }

    /**
     * Track a {@link Content} item being fulfilled using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackFulfilled(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.FULFILLED, content.id, timestamp)
    }

    /**
     * Track a {@link Content} item being fulfilled using an id.
     *
     * @param contentId The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackFulfilled(contentId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.FULFILLED, contentId, timestamp)
    }

    /**
     * Track a {@link Content}'s component being engaged using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackComponentEngaged(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, content.id, timestamp)
    }

    /**
     * Track a {@link Content}'s component being engaged using an id.
     *
     * @param contentId The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackComponentEngaged(contentId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, contentId, timestamp)
    }

    /**
     * Track a {@link Content} item's component being fulfilled using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackComponentFulfilled(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_FULFILLED, content.id, timestamp)
    }

    /**
     * Track a {@link Content}'s component being fulfilled using an id.
     *
     * @param contentId The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackComponentFulfilled(contentId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_FULFILLED, contentId, timestamp)
    }
}