package com.flybits.android.kernel.deserializers;

import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.android.kernel.models.Content;
import com.flybits.android.kernel.models.SurveyMetadata;
import com.flybits.commons.library.deserializations.IDeserializer;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.utils.Utilities;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Iterator;

public class DeserializeContentInstance extends IDeserializer<Content> {

    private static ArrayList<String> languageCodes;
    private Context mContext;

    public DeserializeContentInstance(@NonNull Context context) {
        mContext = context;
        languageCodes = Utilities.getDeviceLanguageCodes(mContext);
    }

    @Override
    public String toJson(Content contentResponse) {
        final JSONObject object = new JSONObject();

        final JSONObject localizationObject = new JSONObject();
        final ArrayList<String> listofCodes = new ArrayList<>();
        for (String code : contentResponse.getNameObject().getListOfSupportedLanguages()) {

            try {
                if (contentResponse.getName(code) != null) {

                    listofCodes.add(code);
                    final JSONObject objectLanguageCode = new JSONObject();
                    objectLanguageCode.put("name", contentResponse.getName(code));
                    localizationObject.put(code, objectLanguageCode);

                    if (contentResponse.getDescription(code) != null) {
                        objectLanguageCode.put("description", contentResponse.getDescription(code));
                        localizationObject.put(code, objectLanguageCode);
                    }
                }
            } catch (JSONException e) {
                Logger.exception("DeserializeContentInstance.toJson", e);
            }
        }

        for (String code : contentResponse.getDescriptionObject().getListOfSupportedLanguages()) {

            try {
                if (!listofCodes.contains(code) && contentResponse.getDescription(code) != null) {
                    final JSONObject objectLanguageCode = new JSONObject();
                    objectLanguageCode.put("description", contentResponse.getDescription(code));
                    localizationObject.put(code, objectLanguageCode);
                }
            } catch (JSONException e) {
                Logger.exception("DeserializeContentInstance.toJson", e);
            }
        }

        try {
            object.put("localizations", localizationObject);
            if (contentResponse.getIcon() != null) {
                object.put("iconUrl", contentResponse.getIcon());
            }
            object.put("templateId", contentResponse.getTemplateId());
            object.put("templateType", contentResponse.getType());
        } catch (JSONException e) {
            Logger.exception("DeserializeContentInstance.toJson", e);
        }

        return object.toString();
    }

    @Override
    public Content fromJson(JSONObject jsonObject) {
        try {
            final String id = jsonObject.getString("id");
            final String templateId = jsonObject.getString("templateId");
            long createdAt = -1;
            if (!jsonObject.isNull("createdAt")) {
                createdAt = jsonObject.getLong("createdAt");
            }
            long modifiedAt = -1;
            if (!jsonObject.isNull("modifiedAt")) {
                modifiedAt = jsonObject.getLong("modifiedAt");
            }

            String iconUrl = null;
            if (!jsonObject.isNull("iconUrl")) {
                iconUrl = jsonObject.getString("iconUrl");
            }

            String type = null;
            if (!jsonObject.isNull("templateType")) {
                type = jsonObject.getString("templateType");
            }

            final JSONObject dataContainer = jsonObject.optJSONObject("content");

            String data = null;
            if (dataContainer != null) {
                data = dataContainer.toString();
            }

            final ArrayList<String> labels = new ArrayList<>();
            if (!jsonObject.isNull("labels")) {
                final JSONArray jsonArray = jsonObject.getJSONArray("labels");
                for (int i = 0; i < jsonArray.length(); i++) {
                    labels.add(jsonArray.getString(i));
                }
            }
            Content instance = new Content(id, templateId, iconUrl, "en",
                    languageCodes.get(0),
                    createdAt, modifiedAt, data, labels, type);

            if (!jsonObject.isNull("localizations")) {

                final JSONObject localizationObject = jsonObject.getJSONObject("localizations");
                final Iterator<String> languages = localizationObject.keys();

                while (languages.hasNext()) {

                    final String language = languages.next();
                    final JSONObject languageObj = localizationObject.getJSONObject(language);
                    if (!languageObj.isNull("description")) {
                        instance.setDescription(language, languageObj.getString("description"));
                    }

                    if (!languageObj.isNull("name")) {
                        instance.setName(language, languageObj.getString("name"));
                    }
                }
            }

            if (!jsonObject.isNull("surveyMetadata")) {
                final JSONObject surveyMetadata = jsonObject.optJSONObject("surveyMetadata");
                final SurveyMetadata obj = new SurveyMetadata(surveyMetadata);
                instance.setMetadata(obj);
            }

            return instance;
        } catch (JSONException e) {
            Logger.exception("DeserializeContentInstance.fromJson", e);
        }
        return null;
    }

    @Override
    public Content fromJson(JSONArray jsonArray) {
        return null;
    }
}
