package com.flybits.android.kernel

import android.content.Context
import com.flybits.android.kernel.models.Content
import com.flybits.commons.library.analytics.Analytics

/**
 * This class is responsible for tracking Experience id with Content analytics. Example of this include:
 * content viewed, content engaged, or content component engaged.
 *
 *  JSON example:
 *  {"dataTypeID":"ctx.flybits.experienceContentAnalytics","value":{"query.viewed.56407410-5764-426B-8561-7AFB1FA2FCBB.942E0543-7A13-4C69-B0D1-7EAF4B6BE42E":true},"timestamp":1582744489}
 *
 * @param context The context of the application.
 */
class ExperienceContentAnalytics(context: Context): ContentAnalytics(context) {

    companion object {
        const val analyticsScope = "experienceContentAnalytics"
    }

    override fun track(action: Action, suffix: String, timestamp: Long) {
        super.track(AnalyticsEvent(analyticsScope, "query.${action.actionName}.$suffix", true, timestamp))
    }

    override fun trackViewed(content: Content, timestamp: Long) {
        track(Action.VIEWED, "${content.id}.${content.experienceId}", timestamp)
    }

    override fun trackEngaged(content: Content, timestamp: Long) {
        track(Action.ENGAGED, "${content.id}.${content.experienceId}", timestamp)
    }

    override fun trackFulfilled(content: Content, timestamp: Long) {
        track(Action.FULFILLED, "${content.id}.${content.experienceId}", timestamp)
    }

    override fun trackComponentEngaged(content: Content, timestamp: Long) {
        track(Action.COMPONENT_ENGAGED, "${content.id}.${content.experienceId}", timestamp)
    }

    override fun trackComponentFulfilled(content: Content, timestamp: Long) {
        track(Action.COMPONENT_FULFILLED, "${content.id}.${content.experienceId}", timestamp)
    }
}