package com.flybits.android.kernel.models;

import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.Nullable;

/**
 * The {@link LocationData} class is used to store location data about {@link Content}. As part of
 * the {@link LocationData} a distance is included that represents the distance between
 * {@link Content} and a specific location.
 */
public class LocationData implements Parcelable {

    private double lat;
    private double lng;
    private double distance;
    private String address;

    /**
     * Default Constructor used to set the latitude, longitude, distance and unit type of a
     * location.
     *
     * @param lat The latitude of the location.
     * @param lng The longitude of the location.
     * @param distance The distance between the given geo-point and the {@link Content} location.
     * @param address The physical address associated to the {@code lat} and {@code lng}
     *                coordinates.
     */
    public LocationData(double lat, double lng, double distance, String address){
        this.lat    = lat;
        this.lng    = lng;
        this.distance   = distance;
        this.address = address;
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code Content}.
     */
    public LocationData(Parcel in) {
        lat = in.readDouble();
        lng = in.readDouble();
        distance = in.readDouble();
        address = in.readString();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeDouble(lat);
        dest.writeDouble(lng);
        dest.writeDouble(distance);
        dest.writeString(address);
    }

    public static final Creator<LocationData> CREATOR = new Creator<LocationData>() {
        @Override
        public LocationData createFromParcel(Parcel in) {
            return new LocationData(in);
        }

        @Override
        public LocationData[] newArray(int size) {
            return new LocationData[size];
        }
    };

    /**
     * Get the address of Location.
     *
     * @return The String representation of the address.
     */
    @Nullable
    public String getAddress(){
        return address;
    }

    /**
     * Gets the distance between the given geo-point and the {@link Content} location.
     *
     * @return the distance between the given geo-point and the {@link Content} location.
     */
    public double getDistance() {
        return distance;
    }

    /**
     * Gets the latitude of the location.
     *
     * @return the latitude of the location.
     */
    public double getLat() {
        return lat;
    }

    /**
     * Gets the longitude of the location.
     *
     * @return the longitude of the location.
     */
    public double getLng() {
        return lng;
    }

    /**
     * The {@link Unit} is an enumerator that represents all of {@link Unit}s that
     * {@link #getDistance()} can be calculated in.
     */
    public enum Unit{
        /**
         * Indicates the unit measured in metres.
         */
        METRES {
            public double toFeet(double distance){return distance*FEET_VALUE;}
            public double toKilometres(double distance){return distance* KILOMETRES_VALUE;}
            public double toMiles(double distance){return distance*MILES_VALUE;}
            public double toMetres(double distance){return distance* METRES_VALUE;}

        },

        /**
         * Indicates the unit measured in feet.
         */
        FEET{
            public double toMetres(double distance){return distance/FEET_VALUE;}
            public double toKilometres(double distance){return toMetres(distance) * KILOMETRES_VALUE;}
            public double toMiles(double distance){return toMetres(distance) * MILES_VALUE;}
            public double toFeet(double distance){return distance;}
        },

        /**
         * Indicates the unit measured in kilometres.
         */
        KILOMETRES {
            public double toMetres(double distance){return distance/ KILOMETRES_VALUE;}
            public double toFeet(double distance){return toMetres(distance) * FEET_VALUE;}
            public double toMiles(double distance){return toMetres(distance) * MILES_VALUE;}
            public double toKilometres(double distance){return distance;}
        },

        /**
         * Indicates the unit measured in miles.
         */
        MILES{
            public double toMetres(double distance){return distance/MILES_VALUE;}
            public double toFeet(double distance){return toMetres(distance) * FEET_VALUE;}
            public double toKilometres(double distance){return toMetres(distance) * KILOMETRES_VALUE;}
            public double toMiles(double distance){return distance;}
        };

        private static final double METRES_VALUE        = 1;
        private static final double FEET_VALUE          = METRES_VALUE * 3.28084;
        private static final double KILOMETRES_VALUE    = METRES_VALUE * 0.001;
        private static final double MILES_VALUE         = METRES_VALUE * 0.000621371;

        /**
         * Calculates the distance into Imperical `feet` measurement unit.
         *
         * @param distance A value to calculate distance in feet.
         * @return The new distance in feet.
         */
        public abstract double toFeet(double distance);

        /**
         * Calculates the distance into Metric `Kilometers` measurement unit.
         *
         * @param distance A value to calculate distance in kilometers.
         * @return The new distance in kilometers.
         */
        public abstract double toKilometres(double distance);

        /**
         * Calculates the distance into Metric `Meters` measurement unit.
         *
         * @param distance A value to calculate distance in meters.
         * @return The new distance in meters.
         */
        public abstract double toMetres(double distance);

        /**
         * Calculates the distance into Metric `Miles` measurement unit.
         *
         * @param distance A value to calculate distance in miles.
         * @return The new distance in miles.
         */
        public abstract double toMiles(double distance);
    }

}
