package com.flybits.android.kernel

import android.content.Context
import com.flybits.android.kernel.models.Content

/**
 * This class is responsible for tracking Push Request id associated to [Content] with Content analytics. Example of this include:
 * content viewed, content engaged, or content component engaged.
 *
 *  JSON example:
 *  {"dataTypeID":"ctx.flybits.pushContentAnalytics","value":{"query.viewed.56407410-5764-426B-8561-7AFB1FA2FCBB.942E0543-7A13-4C69-B0D1-7EAF4B6BE42E":true},"timestamp":1582744489}
 *
 * @param context The context of the application.
 */
class PushContentAnalytics(context: Context) :
    ContentAnalytics(context) {
    companion object {
        const val analyticsScope = "pushContentAnalytics"
    }

    override fun track(action: Action, suffix: String, timestamp: Long) {
        super.track(
            AnalyticsEvent(
                analyticsScope,
                "query.${action.actionName}.$suffix",
                true,
                timestamp
            )
        )
    }

    override fun trackViewed(content: Content, timestamp: Long) {
        track(Action.VIEWED, "${content.id}.${content.pushRequestId}", timestamp)
    }

    override fun trackEngaged(content: Content, timestamp: Long) {
        track(Action.ENGAGED, "${content.id}.${content.pushRequestId}", timestamp)
    }

    override fun trackFulfilled(content: Content, timestamp: Long) {
        track(Action.FULFILLED, "${content.id}.${content.pushRequestId}", timestamp)
    }

    override fun trackComponentEngaged(content: Content, timestamp: Long) {
        track(Action.COMPONENT_ENGAGED, "${content.id}.${content.pushRequestId}", timestamp)
    }

    override fun trackComponentFulfilled(
        content: Content,
        timestamp: Long
    ) {
        track(Action.COMPONENT_FULFILLED, "${content.id}.${content.pushRequestId}", timestamp)
    }
}