package com.flybits.android.kernel

import android.content.Context
import com.flybits.android.kernel.models.Content
import com.flybits.commons.library.analytics.Analytics

/**
 * This class is responsible for tracking Content analytics. Example of this include:
 * content viewed, content engaged, or content component engaged.
 *
 * @param context The context of the application.
 */
open class ContentAnalytics(context: Context): Analytics(context) {

    companion object {
        const val analyticsScope = "contentAnalytics"
    }

    internal enum class Action(val actionName: String){
        VIEWED("viewed")
        , ENGAGED("engaged")
        , FULFILLED("fulfilled")
        , COMPONENT_ENGAGED("componentEngaged")
        , COMPONENT_FULFILLED("componentFulfilled")
    }

    internal open fun track(action: Action, suffix: String, timestamp: Long){
        super.track(AnalyticsEvent(analyticsScope, "query.${action.actionName}.$suffix", true, timestamp))
    }

    /**
     * Track a {@link Content} item being viewed using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    open fun trackViewed(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, content.id, timestamp)
    }

    /**
     * Track a {@link Content} item being viewed using an id.
     *
     * @param suffix The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackViewed(suffix: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, suffix, timestamp)
    }

    /**
     * Track a {@link Content} item being engaged using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    open fun trackEngaged(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, content.id, timestamp)
    }

    /**
     * Track a {@link Content} item being engaged using an id.
     *
     * @param suffix The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackEngaged(suffix: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, suffix, timestamp)
    }

    /**
     * Track a {@link Content} item being fulfilled using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    open fun trackFulfilled(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.FULFILLED, content.id, timestamp)
    }

    /**
     * Track a {@link Content} item being fulfilled using an id.
     *
     * @param suffix The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackFulfilled(suffix: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.FULFILLED, suffix, timestamp)
    }

    /**
     * Track a {@link Content}'s component being engaged using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    open fun trackComponentEngaged(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, content.id, timestamp)
    }

    /**
     * Track a {@link Content}'s component being engaged using an id.
     *
     * @param suffix The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackComponentEngaged(suffix: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, suffix, timestamp)
    }

    /**
     * Track a {@link Content} item's component being fulfilled using a {@link Content} object.
     *
     * @param content The {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    open fun trackComponentFulfilled(content: Content, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_FULFILLED, content.id, timestamp)
    }

    /**
     * Track a {@link Content}'s component being fulfilled using an id.
     *
     * @param suffix The id of the {@link Content} being viewed.
     * @param timestamp Unix timestamp of the event in milliseconds.
     *
     */
    fun trackComponentFulfilled(suffix: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_FULFILLED, suffix, timestamp)
    }
}