package com.flybits.android.kernel.utilities;

import com.flybits.android.kernel.deserializers.FlybitsData;
import com.flybits.android.kernel.models.LocalizedValue;
import com.flybits.android.kernel.models.PagedArray;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.util.ArrayList;

/**
 * The {@link ContentDataSerializer} class is a helper tool for converting a java POJO into
 * JSON.
 */
public class ContentDataSerializer {

    /**
     * Serializes a POJO into content. Both private and public variables will be serialized.
     * See {@link FlybitsData} annotation to customize field names.
     * @param object The JSON to read.
     * @return A serialized object.,
     * TODO: Missing @throws - Fix all comments here
     */
    public static String serialize(Object object) throws FlybitsException {
        return serializeObject(object).toString();
    }

    /**
     * Serializes an object and any sub-objects recursively.
     * @param currentObject The object to be serialized
     * @return JSON serialization of object
     * @throws FlybitsException If there was an issue with serialization
     */
    private static JSONObject serializeObject(Object currentObject) throws FlybitsException {

        JSONObject jsonObject = new JSONObject();

        for (Field f : FieldHelper.getAllFields(currentObject.getClass())) {
            //we need to ignore synthetic members here https://www.eclemma.org/jacoco/trunk/doc/faq.html
            if (!f.isSynthetic()) {
                try {
                    f.setAccessible(true);

                    //Skip creator and content id fields, and those labeled as ignored
                    if (f.getName().equals("CREATOR") || FieldHelper.isContentIdField(f) || FieldHelper.isIgnored(f))
                        continue;

                    String fieldName = FieldHelper.getFieldName(f);

                    //Null check, if localize value ignore otherwise set null
                    if (f.get(currentObject) == null) {
                        if (!LocalizedValue.class.isAssignableFrom(f.getType()))
                            jsonObject.put(fieldName, JSONObject.NULL);
                        continue;
                    }

                    //Paged Arrays
                    if (PagedArray.class.isAssignableFrom(f.getType())) {
                        jsonObject.put(fieldName, serializeArray(((PagedArray) f.get(currentObject)).getList()));
                    }
                    //Unpaged Arrays
                    else if (ArrayList.class.isAssignableFrom(f.getType())) {
                        jsonObject.put(fieldName, serializeArray((ArrayList) f.get(currentObject)));
                    }
                    //Localized Strings
                    else if (LocalizedValue.class.isAssignableFrom(f.getType())) {

                        //Get or check if localization object exists
                        JSONObject localizationsObject = null;
                        try {
                            localizationsObject = jsonObject.getJSONObject("localizations");
                        } catch (JSONException e) {
                        }

                        //Create localizations object if none exists
                        if (localizationsObject == null) {
                            localizationsObject = new JSONObject();
                            jsonObject.put("localizations", localizationsObject);
                        }

                        LocalizedValue value = (LocalizedValue) f.get(currentObject);
                        for (String lang : value.getListOfSupportedLanguages()) {
                            if (!localizationsObject.has(lang))
                                localizationsObject.put(lang, new JSONObject());
                            localizationsObject.optJSONObject(lang).put(fieldName, value.getValue(lang));
                        }
                    }
                    //Normal Primitives
                    else if (FieldHelper.checkIfPrimitiveType(f.getType())) {
                        jsonObject.put(fieldName, f.get(currentObject));
                    }
                    //Objects
                    else
                        jsonObject.put(fieldName, serializeObject(f.get(currentObject)));
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataSerializer.serializeObject", e);
                    throw new FlybitsException("ContentDataSerializer.serializeObject.RELFECTION_ERROR");
                } catch (JSONException e) {
                    Logger.exception("ContentDataSerializer.serializeObject", e);
                    throw new FlybitsException("ContentDataSerializer.serializeObject.JSON_SERIALIZATION_ERROR");
                }
            }
        }

        return jsonObject;
    }

    /**
     * Serializes an array and any objects of primitives it contains
     * @param array The array to be serialized.
     * @return The JSON representation of this array.
     * @throws FlybitsException If there was an issue with serialization.
     */
    private static JSONArray serializeArray(ArrayList array) throws FlybitsException {

        JSONArray jsonArray = new JSONArray();

        for (int i = 0; i < array.size(); i++)
        {
            if (FieldHelper.checkIfPrimitiveType(array.get(i).getClass()))
                jsonArray.put(array.get(i));
            else
                jsonArray.put(serializeObject(array.get(i)));
        }

        return jsonArray;
    }
}
