package com.flybits.android.kernel.models;

import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import com.flybits.android.kernel.db.KernelDatabase;
import com.flybits.android.kernel.deserializers.DeserializeGroup;
import com.flybits.android.kernel.models.results.GroupResult;
import com.flybits.android.kernel.utilities.GroupParameters;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * A {@code Group} is a collection of experiences.
 */
@Entity(tableName = "groups")//Cannot use "group" as tableName since "group" is a reserved key in SQLite.
public class Group implements Parcelable{

    public static final String API      = Experience.API + "/relevantgroups";

    @PrimaryKey
    @NonNull
    private String id;
    private String creatorId;

    private LocalizedValue nameObject;
    private LocalizedValue descriptionObject;
    private long createdAt, modifiedAt;

    private String defaultLanguage;

    //TODO: Confirm if experiences are needed. If not, remove this attribute and its getters/setters.
    @Ignore//@Embedded
    private ArrayList<Experience> experiences;

    /**
     * Default constructor used to allow ROOM to auto-initiate a {@code Group} object from Cache.
     */
    public Group(){}

    /**
     * Constructor that initializes the basic attributes of an {@code Group}.
     * @param id The unique id of this group.
     * @param creatorId The id of the user that created this group.
     * @param experiences A list of experiences contained in this group.
     * @param createdAt The time in seconds when this group was created.
     * @param modifiedAt The time in seconds when this group was last modified.
     * @param defaultLang The default language to use when getting localized strings from this group.
     */
    @Ignore
    public Group(@NonNull String id, @NonNull String creatorId, @NonNull ArrayList<Experience> experiences,
                 long createdAt, long modifiedAt, @NonNull String defaultLang) {
        this.id = id;
        this.creatorId = creatorId;
        this.experiences = experiences;
        this.createdAt = createdAt;
        this.modifiedAt = modifiedAt;
        this.defaultLanguage = defaultLang;

        nameObject = new LocalizedValue(defaultLang, defaultLang);
        descriptionObject = new LocalizedValue(defaultLang, defaultLang);
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code Group}.
     */
    @Ignore
    protected Group(Parcel in) {
        id = in.readString();
        creatorId = in.readString();
        experiences = in.createTypedArrayList(Experience.CREATOR);
        nameObject = in.readParcelable(LocalizedValue.class.getClassLoader());
        descriptionObject = in.readParcelable(LocalizedValue.class.getClassLoader());
        createdAt = in.readLong();
        modifiedAt = in.readLong();
        defaultLanguage = in.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(id);
        dest.writeString(creatorId);
        dest.writeTypedList(experiences);
        dest.writeParcelable(nameObject, flags);
        dest.writeParcelable(descriptionObject, flags);
        dest.writeLong(createdAt);
        dest.writeLong(modifiedAt);
        dest.writeString(defaultLanguage);
    }

    /**
     * Get the time (in milliseconds - epoch) that indicates when this {@code Group} was
     * created.
     *
     * @return time in seconds.
     */
    public long getCreatedAt() {
        return createdAt;
    }

    /**
     * Get the unique identifier representing the user that created this {@code Group}.
     *
     * @return The user identifier of the person who created this {@code Group}.
     */
    public String getCreatorId() {
        return creatorId;
    }

    /**
     * Get the default language set for this {@code Group} object. The default language is a
     * 2-letter code such as "en" or "fr".
     *
     * @return The 2-letter code representing the default language of the {@code Group}, such as
     * "en" or "fr".
     */
    public String getDefaultLanguage(){
        return defaultLanguage;
    }

    /**
     * Get the description of the {@code Group} based on the selected default by the device. If
     * the device does not automatically save the language for whatever reason the default language
     * from the {@link #defaultLanguage} is used.
     *
     * @return The description of the {@code Group}.
     */
    public String getDescription(){
        return descriptionObject.getValue();
    }

    /**
     * Get the description of the {@code Group} based on the {@code languageCode} parameter. If
     * the {@code languageCode} parameter is null or a {@code languageCode} is currently not present
     * in the {@link LocalizedValue} description value then null is returned.
     *
     * TODO: Missing @params
     * @return The description of the {@code Group} based on the {@code languageCode}, if no
     * match is found then null is returned.
     */
    public String getDescription(String languageCode){
        return descriptionObject.getValue(languageCode);
    }

    /**
     * Get the {@link LocalizedValue} of the description of the {@code Group}.
     *
     * @return The description of the {@code Group} through the {@link LocalizedValue}.
     */
    public LocalizedValue getDescriptionObject(){
        return descriptionObject;
    }

    /**
     * Get the {@code Experience}s that are part of this group.
     *
     * @return A list of {@code Experience}s in this group.
     */
    public ArrayList<Experience> getExperiences()
    {
        return experiences;
    }

    /**
     * Get the unique identifier representing this {@code Group}.
     *
     * @return The unique identifier of this {@code Group}.
     */
    public String getId() {
        return id;
    }

    /**
     * Get the time (in milliseconds - epoch) that indicates when this {@code Group} was last
     * modified.
     *
     * @return time in seconds.
     */
    public long getModifiedAt() {
        return modifiedAt;
    }

    /**
     * Get the name of the {@code Group} based on the selected default by the device. If the
     * device does not automatically save the language for whatever reason the default language from
     * the {@link #defaultLanguage} is used.
     *
     * @return The name of the {@code Group}.
     */
    public String getName(){
        return nameObject.getValue();
    }

    /**
     * Get the name of the {@code Group} based on the {@code languageCode} parameter. If
     * the {@code languageCode} parameter is null or a {@code languageCode} is currently not present
     * in the {@link LocalizedValue} name value then null is returned.
     *
     * TODO: Missing @params
     * @return The name of the {@code Group} based on the {@code languageCode}, if no match is
     * found then null is returned.
     */
    public String getName(String languageCode){
        return nameObject.getValue(languageCode);
    }

    /**
     * Get the {@link LocalizedValue} of the name of the {@code Group}.
     *
     * @return The name of the {@code Group} through the {@link LocalizedValue}.
     */
    public LocalizedValue getNameObject(){
        return nameObject;
    }

    /**
     * Set the time in epoch that indicates when this {@code Group} was created.
     *
     * @param createdAt The epoch time that indicates when this {@code Group} was created.
     */
    public void setCreatedAt(long createdAt){
        this.createdAt  = createdAt;
    }

    /**
     * Set the unique identifier of the {@code User} who created this group.
     *
     * @param creatorId The String GUID of the {@code User} who created this {@code Group}.
     */
    public void setCreatorId(String creatorId){
        this.creatorId  = creatorId;
    }

    /**
     * Set the default language for this {@code Group}. The default language is needed in order to
     * define a default language when attempting to retrieve the {@code name} and/or
     * {@code description} of the {@code Group}.
     *
     * @param defaultLanguage The 2-letter code representing the default language of the
     * {@code Group}, such as "en" or "fr".
     */
    public void setDefaultLanguage(String defaultLanguage){
        this.defaultLanguage = defaultLanguage;
    }

    /**
     * Set the {@code description} {@link LocalizedValue} of the {@code Group}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code description}
     * attribute of this {@code Group}. The default language will be used.
     *
     * @param description The description of the {@code Group}.
     */
    public void setDescription(@NonNull String description){
        this.descriptionObject.addValue(defaultLanguage, description);
    }

    /**
     * Set the {@code description} {@link LocalizedValue} of the {@code Group}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code description}
     * attribute of this {@code Group}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param description The description of the {@code Group}.
     */
    public void setDescription(@NonNull String languageCode, @NonNull String description){
        this.descriptionObject.addValue(languageCode, description);
    }

    /**
     * Sets the {@link LocalizedValue} of the description. The {@link LocalizedValue} object is used
     * in order to allow information about this {@code Group} to contain multiple languages and
     * therefore present the data in the language desired by the user of the application.
     *
     * @param value The {@link LocalizedValue} object that defines the description of the
     * {@code Group} in the various available languages.
     */
    public void setDescriptionObject(LocalizedValue value){
        this.descriptionObject  = value;
    }

    /**
     * Set the {@link Experience}s that are associated to this {@code Group}. This attribute may be
     * removed shortly as groups should have access to {@link Experience}s from the mobile SDK.
     *
     * @param experiences The list of {@link Experience}s associated to this {@code Group}.
     */
    public void setExperiences(ArrayList<Experience> experiences){
        this.experiences    = experiences;
    }

    /**
     * Set the unique identifier for the {@code Group}. This identifier is used to delete the
     * {@code Group} as well as retrieve information that may be associated to it.
     *
     * @param id The unique GUID that represents this {@code Group}.
     */
    public void setId(@NonNull String id){
        this.id= id;
    }

    /**
     * Set the time in epoch that indicates when this {@code Group} was last modified.
     *
     * @param modifiedAt The epoch time that indicates when this {@code Group} was last modified.
     */
    public void setModifiedAt(long modifiedAt){
        this.modifiedAt  = modifiedAt;
    }

    /**
     * Set the {@code name} {@link LocalizedValue} of the {@code Group}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code name} attribute of
     * this {@code Group}. The default language will be used.
     *
     * @param name The name of the {@code Group}.
     */
    public void setName(@NonNull String name){
        this.nameObject.addValue(defaultLanguage, name);
    }

    /**
     * Set the {@code name} {@link LocalizedValue} of the {@code Group}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code name} attribute of
     * this {@code Group}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param name The name of the {@code Group}.
     */
    public void setName(@NonNull String languageCode, @NonNull String name){
        this.nameObject.addValue(languageCode, name);
    }

    /**
     * Set the {@link LocalizedValue} of the name. The {@link LocalizedValue} object is used in
     * order to allow information about this {@code Group} to contain multiple languages and
     * therefore present the data in the language desired by the user of the application.
     *
     * @param value The {@link LocalizedValue} object that defines the bane of the {@code Group} in
     *              the various available languages.
     */
    public void setNameObject(LocalizedValue value){
        this.nameObject = value;
    }

    /**
     * Get a list of {@code Groups} based on the {@link GroupParameters} parameter which defines
     * which options should be used to filter the list of returned parameters.
     *
     * @param context The context of the application.
     * @param params The {@link GroupParameters} that indicate filters that should be placed on
     *               the request.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * TODO: Missing @return
     */
    public static GroupResult get(final Context context, final GroupParameters params,
                                  final PagedResultCallback<Group> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final GroupResult result = new GroupResult(context, params, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    DeserializeGroup singleDeserializaer            = new DeserializeGroup(context);
                    DeserializePagedResponse<Group> deserializer    = new DeserializePagedResponse<Group>(singleDeserializaer);
                    final Result<PagedResponse<Group>> getGroups = FlyAway.get(context, API, params, deserializer, "Group.get");

                    //Update the DB for caching/observing purposes
                    if (getGroups.getStatus() == RequestStatus.COMPLETED) {
                        if ((params.getQueryParams().get("offset") == null
                                || params.getQueryParams().get("offset").size() == 0
                                || params.getQueryParams().get("offset").get(0).equals("0"))
                                && (params.getCachingKey() != null)) {

                            List<String> listOfIds  = CommonsDatabase.getDatabase(context).cachingEntryDAO().getIdsByCachingKey("A");
                            KernelDatabase.getDatabase(context).contentDao().deleteByIds(listOfIds);
                            CommonsDatabase.getDatabase(context).cachingEntryDAO().deleteAllByCachingKey(params.getCachingKey());
                        }

                        int maxNumberOfRecordsToSave = (params.getCachingLimit() < getGroups.getResult().getItems().size() + getGroups.getResult().getPagination().getOffset())
                                ? params.getCachingLimit() : getGroups.getResult().getItems().size();

                        if (params.getCachingKey() != null && maxNumberOfRecordsToSave > 0) {
                            ArrayList<CachingEntry> entries = new ArrayList<>();
                            ArrayList<Group> groupToCache = new ArrayList<>();

                            for (int i = 0; i < getGroups.getResult().getItems().size(); i++) {
                                if (i >= maxNumberOfRecordsToSave){
                                    break;
                                }
                                groupToCache.add(getGroups.getResult().getItems().get(i));
                                entries.add(new CachingEntry(params.getCachingKey(), getGroups.getResult().getItems().get(i).getId()));
                            }

                            KernelDatabase.getDatabase(context).groupDao().insert(groupToCache);
                            CommonsDatabase.getDatabase(context).cachingEntryDAO().insert(entries);
                        }
                    }
                    result.setResult(getGroups);
                }catch (final FlybitsException e){
                    result.setResult(new Result<PagedResponse<Group>>(e, "Group.get() failed"));
                }
            }
        });
        return result;
    }

    /**
     * Get a list of {@code Groups} based on the {@link GroupParameters} parameter which defines
     * which options should be used to filter the list of returned parameters.
     *
     * TODO: Missing @return
     * @param mContext The context of the application.
     * @param params The {@link GroupParameters} that indicate filters that should be placed on
     *               the request.
     */
    public static GroupResult get(final Context mContext, final GroupParameters params){
        return get(mContext, params, null);
    }

    /**
     * Delete this {@code Group}.
     *
     * @param mContext The context of the application.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code BasicResult} that contains the network request for deleting a specific
     * {@code Group}.
     */
    public BasicResult delete(final Context mContext, final BasicResultCallback callback){

        if (id == null){
            throw new IllegalArgumentException("You must have an id set to able to delete an Group");
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result deleted = FlyAway.delete(mContext, API, "Group.delete", id);

                    if (deleted.getStatus() == RequestStatus.COMPLETED){
                        KernelDatabase.getDatabase(mContext).groupDao().delete(Group.this);
                    }
                    request.setResult(deleted);
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Group> CREATOR = new Creator<Group>() {
        @Override
        public Group createFromParcel(Parcel in) {
            return new Group(in);
        }

        @Override
        public Group[] newArray(int size) {
            return new Group[size];
        }
    };
}
