package com.flybits.android.kernel.workers

import android.content.Context
import android.content.SharedPreferences
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.flybits.android.kernel.KernelScope
import com.flybits.android.kernel.models.Content
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.internal.db.CommonsDatabase
import com.flybits.internal.db.models.Preference
import com.flybits.internal.models.preferences.FlybitsFavourite
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import java.util.*
import java.util.concurrent.CountDownLatch
import java.util.concurrent.TimeUnit

/**
 * A Worker class that syncs favourites to local database from server
 *
 * @param context The [Context] of the application.
 * @param params Any params that define how the worker behaves
 *
 */

internal const val NUMBER_OF_RUNS = "timesThatFavouritesHasRun"
internal const val FAVOURITE_KEY_OLD = "ContentFavourite"

class FavouritesSyncWorker(context: Context, params: WorkerParameters) : Worker(context, params) {

    override fun doWork(): Result {
        val flybitsFavourite = FlybitsFavourite(applicationContext)
        val preferences = KernelScope.getKernelPreferences(applicationContext)
        return doWork(applicationContext, preferences, flybitsFavourite)
    }

    companion object {
        fun doWork(context : Context, sharedPreferences: SharedPreferences,
                   favourite : FlybitsFavourite) : Result{

            var result = Result.failure()
            val countDownLatch = CountDownLatch(1)
            var job : Job? = null

            val numberOfRuns = sharedPreferences.getInt(NUMBER_OF_RUNS, 0)

            //We only want to run this migration successfully once hence
            if (numberOfRuns == 0){

                //Get the old favourites associated to the wrong key
                favourite.getFromServer(FAVOURITE_KEY_OLD, object :
                    ObjectResultCallback<ArrayList<String>> {

                    override fun onSuccess(oldFavouritesList: ArrayList<String>) {

                        //Get the new favourites associated to the good key
                        favourite.getFromServer(Content.FAVOURITE_KEY, object :
                            ObjectResultCallback<ArrayList<String>> {

                            override fun onSuccess(newFavouritesList: ArrayList<String>) {

                                //Combine the results from the wrong key and right key and remove duplicates
                                oldFavouritesList.addAll(newFavouritesList)
                                val uniqueListOfIds = oldFavouritesList.distinct()

                                //Send the newly combined list of favourites to the good key
                                favourite.add(Content.FAVOURITE_KEY, uniqueListOfIds, object : BasicResultCallback{
                                    override fun onSuccess() {
                                        job = CoroutineScope(Dispatchers.IO).launch {

                                            //Delete all the old favourites
                                            CommonsDatabase.getDatabase(context)
                                                .preferenceDAO()
                                                .deleteByPrefKey(FAVOURITE_KEY_OLD)

                                            //Increase the number of runs so that this process is not run again
                                            sharedPreferences.edit()
                                                .putInt(NUMBER_OF_RUNS, 1)
                                                .apply()

                                            result = Result.success()
                                            countDownLatch.countDown()
                                        }
                                    }

                                    override fun onException(exception: FlybitsException) {
                                        job = CoroutineScope(Dispatchers.IO).launch {
                                            result =  Result.failure()
                                            countDownLatch.countDown()
                                        }
                                    }
                                }, true)
                            }

                            override fun onException(exception: FlybitsException) {
                                job = CoroutineScope(Dispatchers.IO).launch {
                                    result =  Result.failure()
                                    countDownLatch.countDown()
                                }
                            }
                        });
                    }
                    override fun onException(exception: FlybitsException) {
                        job = CoroutineScope(Dispatchers.IO).launch {
                            result = Result.failure()
                            countDownLatch.countDown()
                        }
                    }
                })

            }else{

                //Once the migration has taken place do the regular favs syncing - same as before
                favourite.getFromServer(Content.FAVOURITE_KEY, object :
                    ObjectResultCallback<ArrayList<String>> {

                    override fun onSuccess(items: ArrayList<String>) {

                        job = CoroutineScope(Dispatchers.IO).launch {

                            val list = items.distinct().map {
                                Preference(
                                    Content.FAVOURITE_KEY,
                                    it
                                )
                            }

                            val preferenceDAO = CommonsDatabase.getDatabase(context).preferenceDAO()
                            preferenceDAO.deleteByPrefKey(Content.FAVOURITE_KEY)
                            preferenceDAO.insert(list)

                            result =  Result.success()
                            countDownLatch.countDown()
                        }
                    }

                    override fun onException(exception: FlybitsException) {
                        job = CoroutineScope(Dispatchers.Default).launch {
                            result =  Result.failure()
                            countDownLatch.countDown()
                            return@launch
                        }
                    }
                })
            }

            return try {
                if(!countDownLatch.await(10, TimeUnit.SECONDS)) {
                    job?.cancel()
                    Result.failure()
                } else {
                    result
                }
            } catch ( e : InterruptedException){
                Result.retry()
            } catch (e : java.lang.Exception){
                Result.failure()
            }
        }
    }
}