package com.flybits.android.kernel


import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.PeriodicWorkRequest
import androidx.work.WorkManager
import com.flybits.android.kernel.db.KernelDatabase
import com.flybits.android.kernel.workers.FavouritesSyncWorker
import com.flybits.commons.library.api.FlybitsScope
import com.flybits.commons.library.models.User
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

class KernelScope private constructor() :
    FlybitsScope("Kernel") {

    private var enableFavouritesSync: Boolean = true

    constructor(enableFavouritesSync: Boolean = true) : this() {
        this.enableFavouritesSync = enableFavouritesSync
    }

    override fun onConnected(context: Context, user: User) {
        clearDatabaseAsync(context) //In case data is lingering from previous session and onDisconnected() wasn't called
        if (user.isOptedIn) {
            scheduleWorkers()
        }
    }

    override fun onDisconnected(context: Context, s: String) {
        cancelWorkers()
        clearDatabaseAsync(context)
    }

    override fun onAccountDestroyed(context: Context, s: String) {
        clearDatabaseAsync(context)
    }

    override fun onOptedStateChange(context: Context, optedState: Boolean) {
        if (!optedState) {
            clearDatabaseAsync(context)
            cancelWorkers()

        } else {
            scheduleWorkers()
        }
    }

    /**
     * Clear the kernel databases on a worker thread.
     *
     * @param context Context associated with the application.
     */
    internal fun clearDatabaseAsync(context: Context) {
        Executors.newSingleThreadExecutor().execute { clearDatabase(context) }
    }

    private fun scheduleWorkers() {
        if (enableFavouritesSync) {
            val taskGetPluginsRepeat =
                PeriodicWorkRequest.Builder(FavouritesSyncWorker::class.java, 1, TimeUnit.DAYS)
                    .addTag(WORKER_NAME_CONTENT_FAVS)
                    .build()
            WorkManager.getInstance().enqueueUniquePeriodicWork(
                WORKER_NAME_CONTENT_FAVS,
                ExistingPeriodicWorkPolicy.REPLACE,
                taskGetPluginsRepeat
            )
        }
    }

    private fun cancelWorkers() {
        if (enableFavouritesSync) {
            WorkManager.getInstance().cancelUniqueWork(WORKER_NAME_CONTENT_FAVS)
        }
    }

    /**
     * Clear the kernel databases. Warning, this method will run on the caller thread.
     * See [KernelScope.clearDatabaseAsync] for asynchronous alternative.
     *
     * @param context Context associated with the application.
     */
    internal fun clearDatabase(context: Context) {
        val db = KernelDatabase.getDatabase(context)
        db.contentDao().clear()
        db.groupDao().clear()
    }

    companion object {

        internal const val WORKER_NAME_CONTENT_FAVS = "Get_Content_Favs"
        const val ROOT = "/kernel"

        @JvmField
        val SCOPE = KernelScope()

        @JvmField
        val SCOPE_DISABLE_FAV = KernelScope(false)

        /**
         * Get the default shared preferences for the application.
         *
         * @param context The context of the application.
         * @return The default SharedPreferences object for the application.
         */
        fun getKernelPreferences(context: Context): SharedPreferences {
            return context.getSharedPreferences("FLYBITS_PREF_KERNEL", MODE_PRIVATE)
        }
    }
}
