package com.flybits.android.kernel.utilities;

import android.content.Context;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.android.kernel.models.LocalizedValue;
import com.flybits.android.kernel.models.LocationData;
import com.flybits.android.kernel.models.PagedArray;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;

/**
 * The {@link ContentDataDeserializer} class is a helper tool for converting the received JSON data
 * response when retrieving content data, into a POJO class.
 */
public class ContentDataDeserializer {

    /**
     * Deserializes content data in JSON for into a POJO object defined by a class object.
     *
     * @param context The context of the application.
     * @param instanceId The parent content instance id for this data (used for PagedArrays).
     * @param model      The POJO class to deserialize into.
     * @param json       The JSON to read.
     * @param <T>        The POJO class that will be returned.
     * @return An instantiated object based on the model given and json
     * @throws FlybitsException in the event that the {@code model} does not match the {@code json}.
     */
    public static <T> T deserialize(Context context, String instanceId, @NonNull Class model, String json) throws FlybitsException {
        Constructor<T> contentDataConstructor = null;
        try {
            contentDataConstructor = model.getDeclaredConstructor();
            contentDataConstructor.setAccessible(true);
            T contentDataObj = contentDataConstructor.newInstance();

            JSONObject object = new JSONObject(json);
            deserializeObject(context, instanceId, model, contentDataObj, object, "");

            return contentDataObj;
        } catch (NoSuchMethodException e) {
            Logger.exception("ContentDataDeserializer.deserialize", e);
            throw new FlybitsException("ContentDataDeserializer.CANNOT_INSTANTIATE_OBJ");
        } catch (InvocationTargetException e) {
            Logger.exception("ContentDataDeserializer.deserialize", e);
            throw new FlybitsException("ContentDataDeserializer.CANNOT_INSTANTIATE_OBJ");
        } catch (IllegalAccessException e) {
            Logger.exception("ContentDataDeserializer.deserialize", e);
            throw new FlybitsException("ContentDataDeserializer.CANNOT_INSTANTIATE_OBJ");
        } catch (InstantiationException e) {
            Logger.exception("ContentDataDeserializer.deserialize", e);
            throw new FlybitsException("ContentDataDeserializer.CANNOT_INSTANTIATE_OBJ");
        } catch (NullPointerException e) {
            Logger.exception("ContentDataDeserializer.deserialize", e);
            throw new FlybitsException("ContentDataDeserializer.CANNOT_INSTANTIATE_OBJ");
        } catch (JSONException e) {
            Logger.exception("ContentDataDeserializer.deserialize", e);
            throw new FlybitsException("ContentDataDeserializer.MALFORMED_JSON");
        }
    }

    static <T> void deserializeObject(Context context, String instanceId, Class model,
                                      T contentDataObj, JSONObject object, String pathSoFar) throws FlybitsException {
        for (Field f : model.getDeclaredFields()) {
            //The JSON doesn't have this, leave it null;
            if (!LocalizedValue.class.isAssignableFrom(f.getType()) && !object.has(f.getName()))
                continue;

            //Get the type
            Class objectType = f.getType();

            String currentPath = pathSoFar + (pathSoFar.equals("") ? "" : ".") + f.getName();

            //Paged Arrays
            if (PagedArray.class.isAssignableFrom(f.getType())) {
                JSONObject jsonPagingObject = object.optJSONObject(f.getName() + ".pagination");
                JSONArray jsonArray = object.optJSONArray(f.getName());
                if (jsonPagingObject != null && jsonArray != null) {
                    Constructor paginationConstructor;
                    try {
                        //Load the paging portion
                        long totalRecords = jsonPagingObject.getLong("totalRecords");
                        long limit = jsonPagingObject.getLong("limit");
                        long offset = jsonPagingObject.getLong("offset");

                        //Load the array
                        ParameterizedType genericTypeClazz = (ParameterizedType) f.getGenericType();

                        if (genericTypeClazz.getActualTypeArguments().length != 1)
                            continue;

                        Class arrayObjectType = (Class) genericTypeClazz.getActualTypeArguments()[0];
                        ArrayList arrayObj = deserializeArray(context, instanceId, arrayObjectType, jsonArray, currentPath);

                        //Instantiate
                        paginationConstructor = f.getType().getDeclaredConstructor(String.class, String.class, Class.class, Long.TYPE, Long.TYPE, Long.TYPE, ArrayList.class);
                        paginationConstructor.setAccessible(true);
                        PagedArray pagedArray = (PagedArray) paginationConstructor.newInstance(instanceId, currentPath, model, totalRecords, limit, offset, arrayObj);
                        f.set(contentDataObj, pagedArray);
                    } catch (NoSuchMethodException e) {
                        Logger.exception("ContentDataDeserializer.deserializePageObject", e);
                        throw new FlybitsException("ContentDataDeserializer.deserializePageObject.CANNOT_INSTANTIATE_OBJ");
                    } catch (InvocationTargetException e) {
                        Logger.exception("ContentDataDeserializer.deserializePageObject", e);
                        throw new FlybitsException("ContentDataDeserializer.deserializePageObject.CANNOT_INSTANTIATE_OBJ");
                    } catch (IllegalAccessException e) {
                        Logger.exception("ContentDataDeserializer.deserializePageObject", e);
                        throw new FlybitsException("ContentDataDeserializer.deserializePageObject.CANNOT_INSTANTIATE_OBJ");
                    } catch (InstantiationException e) {
                        Logger.exception("ContentDataDeserializer.deserializePageObject", e);
                        throw new FlybitsException("ContentDataDeserializer.deserializePageObject.CANNOT_INSTANTIATE_OBJ");
                    } catch (JSONException e) {
                        Logger.exception("ContentDataDeserializer.deserializePageObject", e);
                        throw new FlybitsException("ContentDataDeserializer.deserializePageObject.MALFORMED_JSON");
                    }
                }
            }
            //Unpaged Arrays (make a fake paged array)
            else if (object.optJSONArray(f.getName()) != null) {
                Constructor paginationConstructor = null;
                try {
                    //Load the array
                    ArrayList<Object> objectArray = new ArrayList<>();
                    ParameterizedType genericTypeClazz = (ParameterizedType) f.getGenericType();

                    if (genericTypeClazz.getActualTypeArguments().length != 1)
                        continue;

                    Class arrayObjectType = (Class) genericTypeClazz.getActualTypeArguments()[0];
                    ArrayList arrayObj = deserializeArray(context, instanceId, arrayObjectType, object.optJSONArray(f.getName()), currentPath);
                    if (List.class.isAssignableFrom(f.getType()))
                        f.set(contentDataObj, arrayObj);
                    else
                        throw new FlybitsException("ContentDataDeserializer.deserializeArrayObject.NOT_PAGEDARRAY_OR_LIST");
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataDeserializer.deserializeArrayObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeArrayObject.CANNOT_INSTANTIATE_OBJ");
                }
            }
            //Localized Strings
            else if (LocalizedValue.class.isAssignableFrom(f.getType())) {
                JSONObject jsonLocalizationObject = object.optJSONObject("localizations");

                try {
                    if (jsonLocalizationObject != null) {
                        HashMap<String, String> values = new LinkedHashMap<>();
                        Iterator<?> keys = jsonLocalizationObject.keys();

                        while (keys.hasNext()) {
                            String key = (String) keys.next();
                            if (!jsonLocalizationObject.getJSONObject(key).isNull(f.getName()))
                                values.put(key, jsonLocalizationObject.getJSONObject(key).getString(f.getName()));
                        }

                        ArrayList<String> languages = SharedElements.INSTANCE.getEnabledLanguagesAsArray(context);
                        String defaultLanguage = (languages.size() > 0 && !languages.get(0).isEmpty()) ? languages.get(0) : "en";

                        LocalizedValue value = new LocalizedValue("en", defaultLanguage); //TODO: FIGURE THE DEFAULTS OUT
                        for (String key : values.keySet()) {
                            value.addValue(key, values.get(key));
                        }

                        //Instantiate
                        f.set(contentDataObj, value);

                    }
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataDeserializer.deserializeLocalizationObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeLocalizationObject.CANNOT_INSTANTIATE_OBJ");
                } catch (JSONException e) {
                    Logger.exception("ContentDataDeserializer.deserializeLocalizationObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeLocalizationObject.MALFORMED_JSON");
                }

            }
            //Location Strings
            else if (LocationData.class.isAssignableFrom(f.getType())) {
                JSONObject jsonLocationObject = object.optJSONObject(f.getName());
                try {
                    if (jsonLocationObject != null) {

                        double lat = 0.0, lng = 0.0, distance = -1.0;
                        String address = null;

                        HashMap<String, String> values = new LinkedHashMap<>();
                        Iterator<?> keys = jsonLocationObject.keys();

                        while (keys.hasNext()) {
                            String key = (String) keys.next();
                            if (key.equalsIgnoreCase("lat")) {
                                lat = jsonLocationObject.getDouble("lat");
                            }
                            if (key.equalsIgnoreCase("lng")) {
                                lng = jsonLocationObject.getDouble("lng");
                            }
                            if (key.equalsIgnoreCase("metadata")) {
                                JSONObject metadataObj = jsonLocationObject.getJSONObject("metadata");
                                if (!metadataObj.isNull("address")){
                                    address = metadataObj.getString("address");
                                }
                            }
                        }
                        if (!object.isNull("___" + f.getName() + "_distance")) {
                            distance = object.getDouble("___" + f.getName() + "_distance");
                        }
                        //Instantiate
                        f.set(contentDataObj, new LocationData(lat, lng, distance, address));
                    }
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataDeserializer.deserializeLocalizationObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeLocalizationObject.CANNOT_INSTANTIATE_OBJ");
                } catch (JSONException e) {
                    Logger.exception("ContentDataDeserializer.deserializeLocalizationObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeLocalizationObject.MALFORMED_JSON");
                }

            }
            //Normal Primitives
            else if (checkIfPrimitiveType(f.getType())) {
                try {
                    Object jsonObject = object.get(f.getName());
                    Class jsonType = jsonObject.getClass();

                    if (objectType.getName().equals("java.lang.Float") && jsonObject instanceof Double)
                        f.set(contentDataObj, ((Double) jsonObject));
                    else if (objectType.getName().equals("java.lang.Double") && jsonObject instanceof Float)
                        f.set(contentDataObj, ((Float) jsonObject).doubleValue());
                    else if (objectType.getName().equals("java.lang.Integer") && jsonObject instanceof Long)
                        f.set(contentDataObj, ((Long) jsonObject).intValue());
                    else if (objectType.getName().equals("java.lang.Long") && jsonObject instanceof Integer)
                        f.set(contentDataObj, ((Integer) jsonObject).longValue());
                    else
                        f.set(contentDataObj, jsonObject);
                } catch (IllegalArgumentException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException(String.format(Locale.getDefault(), "ContentDataDeserializer.deserializeObject.JSON_POJO_DIFFER: " +
                            "Double check that the template and the POJO match! Field Name: %s", f.getName()));
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.CANNOT_SET_PRIMITIVE");
                } catch (JSONException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.MALFORMED_JSON");
                }
            }
            //Objects
            else if (object.optJSONObject(f.getName()) != null) {
                Constructor objConstructor = null;
                try {
                    objConstructor = f.getType().getDeclaredConstructor();
                    objConstructor.setAccessible(true);
                    Object objInstantiated = objConstructor.newInstance();
                    deserializeObject(context, instanceId, objInstantiated.getClass(),
                            objInstantiated, object.getJSONObject(f.getName()), currentPath);
                    f.set(contentDataObj, objInstantiated);
                } catch (NoSuchMethodException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.CANNOT_INSTANTIATE_OBJ");
                } catch (InvocationTargetException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.CANNOT_INSTANTIATE_OBJ");
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.CANNOT_INSTANTIATE_OBJ");
                } catch (InstantiationException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.CANNOT_INSTANTIATE_OBJ");
                } catch (JSONException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.MALFORMED_JSON");
                }
            }
        }
    }

    private static ArrayList deserializeArray(Context context, String instanceId, Class model, JSONArray array, String currentPath) throws FlybitsException {

        ArrayList list = new ArrayList();

        //Array of primitives
        if (checkIfPrimitiveType(model)) {
            for (int i = 0; i < array.length(); i++) {

                String arrayPath = currentPath + "[" + i + "]";

                try {
                    Object jsonObject = array.get(i);
                    Class jsonType = jsonObject.getClass();

                    if (model.getName().equals("java.lang.Float") && jsonObject instanceof Double)
                        list.add(((Double) jsonObject).floatValue());
                    else if (model.getName().equals("java.lang.Double") && jsonObject instanceof Float)
                        list.add(((Float) jsonObject).doubleValue());
                    else if (model.getName().equals("java.lang.Integer") && jsonObject instanceof Long)
                        list.add(((Long) jsonObject).intValue());
                    else if (model.getName().equals("java.lang.Long") && jsonObject instanceof Integer)
                        list.add(((Integer) jsonObject).longValue());
                    else
                        list.add(jsonObject);
                } catch (JSONException e) {
                    Logger.exception("ContentDataDeserializer.deserializeObject", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeObject.MALFORMED_JSON");
                }
            }
        }
        //Array of objects
        else {
            for (int i = 0; i < array.length(); i++) {

                String arrayPath = currentPath + "[" + i + "]";

                try {
                    JSONObject jsonObj = array.getJSONObject(i);

                    Constructor objConstructor = null;
                    objConstructor = model.getDeclaredConstructor();
                    objConstructor.setAccessible(true);
                    Object objInstantiated = objConstructor.newInstance();
                    deserializeObject(context, instanceId, objInstantiated.getClass(), objInstantiated, jsonObj, arrayPath);
                    list.add(objInstantiated);
                } catch (NoSuchMethodException e) {
                    Logger.exception("ContentDataDeserializer.deserializeArray", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeArray.CANNOT_INSTANTIATE_OBJ");
                } catch (InvocationTargetException e) {
                    Logger.exception("ContentDataDeserializer.deserializeArray", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeArray.CANNOT_INSTANTIATE_OBJ");
                } catch (IllegalAccessException e) {
                    Logger.exception("ContentDataDeserializer.deserializeArray", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeArray.CANNOT_INSTANTIATE_OBJ");
                } catch (InstantiationException e) {
                    Logger.exception("ContentDataDeserializer.deserializeArray", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeArray.CANNOT_INSTANTIATE_OBJ");
                } catch (JSONException e) {
                    Logger.exception("ContentDataDeserializer.deserializeArray", e);
                    throw new FlybitsException("ContentDataDeserializer.deserializeArray.MALFORMED_JSON");
                }
            }
        }

        return list;
    }

    private static boolean checkIfPrimitiveType(Class type) {
        return type.getName().equals("java.lang.String") ||
                type.getName().equals("java.lang.Boolean") || type.getName().equals("boolean") ||
                type.getName().equals("java.lang.Float") || type.getName().equals("float") ||
                type.getName().equals("java.lang.Double") || type.getName().equals("double") ||
                type.getName().equals("java.lang.Integer") || type.getName().equals("int") ||
                type.getName().equals("java.lang.Long") || type.getName().equals("long");
    }

    /**
     * Extracts the paged array by path from a deserialized Content data object. This is used by
     * PagedArray as the whole structure (sans everything but the array) is returned by the API.
     *
     * @param object    The deserialized content data which has been paged.
     * @param fieldPath The path to the PagedArray field within this object.
     * @param <T>       The type contained within the array.
     * @return Returns the paged array at the fieldpath.
     * TODO: Missing @throws
     */
    public static <T extends Parcelable> PagedArray<T> extractPagedArray(Object object, String fieldPath) throws FlybitsException {

        //We are at the correct level
        if (!fieldPath.contains(".")) {
            try {
                Field f = object.getClass().getDeclaredField(fieldPath);
                if (f != null && PagedArray.class.isAssignableFrom(f.getType())) {
                    PagedArray<T> array = (PagedArray<T>) f.get(object);
                    return array;
                }
            } catch (NoSuchFieldException e) {
                throw new FlybitsException("Reflection Error");
            } catch (IllegalAccessException e) {
                throw new FlybitsException("Reflection Error");
            }
        }

        String fieldName = fieldPath.substring(0, fieldPath.indexOf("."));

        try {
            Field f = object.getClass().getDeclaredField(fieldName);
            if (f != null && !f.getType().isPrimitive() && !f.getType().isArray()) {
                Object o = f.get(object);
                return extractPagedArray(o, fieldPath.substring(fieldPath.indexOf(".") + 1));
            }
        } catch (NoSuchFieldException e) {
            throw new FlybitsException("Reflection Error");
        } catch (IllegalAccessException e) {
            throw new FlybitsException("Reflection Error");
        }

        return null;
    }
}
