package com.flybits.android.kernel.models;

import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.flybits.android.kernel.api.FlyContentData;
import com.flybits.android.kernel.db.KernelDatabase;
import com.flybits.android.kernel.deserializers.DeserializeContentDatum;
import com.flybits.android.kernel.deserializers.DeserializeContentInstance;
import com.flybits.android.kernel.models.internal.ContentDataResponse;
import com.flybits.android.kernel.models.results.ContentResult;
import com.flybits.android.kernel.utilities.ContentDataParameters;
import com.flybits.android.kernel.utilities.ContentDataSerializer;
import com.flybits.android.kernel.utilities.ContentParameters;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ListResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ListResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializeListResponse;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.JsonParser;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.models.internal.SortByEnumeratable;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;
import com.flybits.internal.db.models.Preference;
import com.flybits.internal.models.preferences.FlybitsFavourite;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * An {@code Content} is a single instance of Content that is based off of an {@code ContentTemplate}. It can
 * have it's own name and description. Each {@code Content} has it's own {@code ContentData} that can be
 * requested.
 */
@Entity(tableName = "content")
public class Content implements Parcelable {

    public static final String FAVOURITE_KEY                   = "ContentFavourite";

    public static final String API_CONTENT_RELEVANT_INSTANCES   = Experience.API+"/contents";
    public static final String API_CONTENT_INSTANCE             = "/kernel/content/instances";
    public static final String API_CONTENT_INSTANCE_WITHDATA    = API_CONTENT_INSTANCE+"/%s?data=true"; //TODO: Create a get that returns ALL instances (not just relevant)

    @PrimaryKey
    @NonNull
    private String id;
    private String templateId;

    private LocalizedValue nameObject;
    private LocalizedValue descriptionObject;

    private long createdAt;
    private long modifiedAt;
    private String icon;
    private String dataAsJson;
    private String type;
    public int sequence;

    private JsonParser metadata;

    private boolean isFavourite;

    @Ignore
    private Parcelable deserializedObject;

    private ArrayList<String> labels;

    /**
     * All possible values that the server can sort request content by. Used by {@code QueryBuilder}.
     */
    public enum SortBy implements SortByEnumeratable{

        /**
         * Sort by the Content's priority set by Experience Studio.
         */
        PRIORITY("index"),

        /**
         * Sort by which Content had it's rule evaluate last.
         */
        EVALUATED_AT("evaluatedAt"),

        /**
         * Sort by the time the Content was created.
         */
        CREATED_AT("createdAt"),

        /**
         * Sort by the time the Content was last modified.
         */
        MODIFIED_AT("modifiedAt");

        private String value;

        SortBy(String value)
        {
            this.value = value;
        }

        @Override
        public String toString() {
            return value;
        }

        @Override
        public String getValue() {
            return toString();
        }

        /**
         * Get the {@code SortBy} enum value corresponding to an String representation.
         *
         * @param key the String representation of the {@code SortBy} enum.
         *
         * @return The {@code SortBy} enum for the String representation.
         */
        public static SortBy fromValue(String key) {
            for(SortBy type : SortBy.values()) {
                if(type.getValue().equalsIgnoreCase(key)) {
                    return type;
                }
            }
            return CREATED_AT;
        }
    }

    /**
     * Default constructor used to allow ROOM to auto-initiate a {@code Content} object from Cache.
     */
    public Content(){}

    @Ignore
    @Deprecated
    private Content(@NonNull String id, @NonNull String templateId, String iconUrl, @NonNull String defaultLanguage,
                    @NonNull String defaultDeviceLangauge, long createdAt, long modifiedAt, ArrayList<String> labels){
        this(id, templateId, iconUrl, defaultLanguage, defaultDeviceLangauge, createdAt, modifiedAt, labels, "");
    }

    @Ignore
    private Content(@NonNull String id, @NonNull String templateId, String iconUrl, @NonNull String defaultLanguage,
                    @NonNull String defaultDeviceLangauge, long createdAt, long modifiedAt, ArrayList<String> labels,
                    String type){
        this.id             = id;
        this.templateId     = templateId;
        this.icon           = iconUrl;

        this.createdAt      = createdAt;
        this.modifiedAt     = modifiedAt;

        nameObject          = new LocalizedValue(defaultLanguage, defaultDeviceLangauge);
        descriptionObject   = new LocalizedValue(defaultLanguage, defaultDeviceLangauge);
        this.labels         = labels;
        this.type           = type;
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code Content}.
     */
    @Ignore
    protected Content(Parcel in) {
        this(in.readString(), in.readString(), in.readString(), in.readString(), in.readString(),
                in.readLong(), in.readLong(), in.createStringArrayList(), in.readString());

        nameObject          = in.readParcelable(LocalizedValue.class.getClassLoader());
        descriptionObject   = in.readParcelable(LocalizedValue.class.getClassLoader());


        try
        {
            metadata = new SurveyMetadata(new JSONObject(in.readString()));
        }
        catch (JSONException e)
        {
            e.printStackTrace();
        }

        if (in.readInt() == 1){
            dataAsJson      = in.readString();
        }

        if (in.readInt() == 1){
            deserializedObject  = in.readParcelable(Parcelable.class.getClassLoader());
        }
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(id);
        dest.writeString(templateId);
        dest.writeString(icon);
        dest.writeString(nameObject.getEntityDefaultLanguage());
        dest.writeString(nameObject.getDeviceDefaultLanguage());

        dest.writeLong(createdAt);
        dest.writeLong(modifiedAt);

        dest.writeStringList(labels);
        dest.writeString(type);

        dest.writeParcelable(nameObject, flags);
        dest.writeParcelable(descriptionObject, flags);

        dest.writeString(metadata != null ? metadata.toJson().toString() : "");

        dest.writeInt(dataAsJson != null ? 1 : 0);
        if (dataAsJson != null){
            dest.writeString(dataAsJson);
        }

        dest.writeInt(deserializedObject != null ? 1 : 0);
        if (deserializedObject != null){
            dest.writeParcelable(deserializedObject, flags);
        }
    }
    /**
     * Default constructor that initiates the basic attributes of an {@code Content}.
     *
     * @param id The unique identifier for the {@code Content}.
     * @param templateId The unique identifier representing the {@code ContentTemplate} this
     *                   instance is based off of.
     * @param iconUrl The url where the icon is stored.
     * @param defaultLanguage The default language that should be used to represent this
     *                        {@code Content}.
     * @param defaultDeviceLangauge The default language set by the device.
     * @param createdAt The time (in seconds - epoch) that indicates when this {@code Content}
     *                  was created.
     * @param modifiedAt The time (in seconds - epoch) that indicates when this {@code Content}
     *                  was last modified.
     * @param dataJson The JSON representation of data field for {@code Content}.
     * @param labels The list of labels associated to the {@code Content}.
     * @deprecated You should use {@link #Content(String, String, String, String, String, long, long, String, ArrayList, String)}
     */
    @Deprecated
    public Content(@NonNull String id, @NonNull String templateId, String iconUrl, @NonNull String defaultLanguage,
                   @NonNull String defaultDeviceLangauge, long createdAt, long modifiedAt, @NonNull String dataJson,
                   ArrayList<String> labels) {

        this(id, templateId, iconUrl, defaultLanguage, defaultDeviceLangauge, createdAt, modifiedAt, labels, "");
        dataAsJson = dataJson;
    }

    /**
     * Default constructor that initiates the basic attributes of an {@code Content}.
     *
     * @param id The unique identifier for the {@code Content}.
     * @param templateId The unique identifier representing the {@code ContentTemplate} this
     *                   instance is based off of.
     * @param iconUrl The url where the icon is stored.
     * @param defaultLanguage The default language that should be used to represent this
     *                        {@code Content}.
     * @param defaultDeviceLangauge The default language set by the device.
     * @param createdAt The time (in seconds - epoch) that indicates when this {@code Content}
     *                  was created.
     * @param modifiedAt The time (in seconds - epoch) that indicates when this {@code Content}
     *                  was last modified.
     * @param dataJson The JSON representation of data field for {@code Content}.
     * @param labels The list of labels associated to the {@code Content}.
     * @param type The user-set identifier for a Content Template
     */
    public Content(@NonNull String id, @NonNull String templateId, String iconUrl, @NonNull String defaultLanguage,
                   @NonNull String defaultDeviceLangauge, long createdAt, long modifiedAt, @NonNull String dataJson,
                   ArrayList<String> labels, String type) {

        this(id, templateId, iconUrl, defaultLanguage, defaultDeviceLangauge, createdAt, modifiedAt, labels, type);
        dataAsJson = dataJson;
    }

    private void changeFavourite(Context context, boolean isFavourite, BasicResultCallback callback){

        FlybitsFavourite favourite = new FlybitsFavourite(context);
        if (callback == null) {
            callback = new BasicResultCallback() {
                @Override
                public void onSuccess() {

                }

                @Override
                public void onException(FlybitsException exception) {

                }
            };
        }

        if (isFavourite) {
            favourite.add(FAVOURITE_KEY, getId(), callback);
        }else{
            favourite.remove(FAVOURITE_KEY, getId(), callback);
        }
    }

    /**
     * Get the time (in seconds - epoch) that indicates when this {@code Content} was
     * created.
     * @return time in seconds.
     */
    public long getCreatedAt()
    {
        return createdAt;
    }

    /**
     * Get the data associated to this {@code Content} as a String variable.
     *
     * @return The String representation of the {@code Content}\'s data object.
     */
    public String getDataAsJson(){
        return dataAsJson;
    }

    /**
     * Get the description of the {@code Content} based on the selected default by the device. If the
     * device does not automatically save the language for whatever reason the default language from
     * the {@code #defaultLanguage} is used.
     *
     * @return The description of the {@code Content}.
     */
    public String getDescription()
    {
        return descriptionObject.getValue();
    }

    /**
     * Get the description of the {@code Content} based on the language given.
     *
     * @param langauge The language to get the description in.
     * @return The description of the {@code Content}.
     */
    public String getDescription(String langauge)
    {
        return descriptionObject.getValue(langauge);
    }

    /**
     * Get the {@link LocalizedValue} of the description of the {@code Content}.
     *
     * @return The description of the {@code Content} through the {@link LocalizedValue}.
     */
    public LocalizedValue getDescriptionObject(){
        return descriptionObject;
    }

    /**
     * Get the icon url for the {@code Content}.
     *
     * @return The url where the icon for this {@code Content} is stored. If no icon is set then
     * {@code null} will be returned.
     */
    @Nullable
    public String getIcon() { return icon; }

    /**
     * Get the unique identifier representing this {@code Content}.
     *
     * @return The unique identifier of this {@code Content}.
     */
    @NonNull
    public String getId()
    {
        return id;
    }

    /**
     * Indicates whether or not this {@code Content} is marked as a favourite.
     *
     * @return true if the {@code Content} was saved as a favourite, false otherwise.
     */
    public boolean getIsFavourite(){
        return isFavourite;
    }

    /**
     * Get the list of labels that are associated to this piece of {@code Content}.
     *
     * @return The list of labels that are associated to this piece of {@code Content}.
     */
    public ArrayList<String> getLabels(){
        if (labels == null){
            labels  = new ArrayList<>();
        }
        return labels;
    }

    /**
     * Get the time (in seconds - epoch) that indicates when this {@code Content} was
     * last modified.
     * @return time in seconds.
     */
    public long getModifiedAt()
    {
        return modifiedAt;
    }

    /**
     * Get the name of the {@code Content} based on the selected default by the device. If the
     * device does not automatically save the language for whatever reason the default language from
     * the {@code #defaultLanguage} is used.
     *
     * @return The name of the {@code Content}.
     */
    public String getName() { return nameObject.getValue(); }

    /**
     * Get the name of the {@code Content} based on the language given.
     *
     * @param language The language to get the name in.
     * @return The name of the {@code Content}.
     */
    public String getName(String language) { return nameObject.getValue(language); }

    /**
     * Get the {@link LocalizedValue} of the name of the {@code Content}.
     *
     * @return The name of the {@code Content} through the {@link LocalizedValue}.
     */
    public LocalizedValue getNameObject(){
        return nameObject;
    }

    /**
     * Get the unique identifier of the {@code ContentTemplate} this instance is based off of.
     * @return The unique identifier of the {@code ContentTemplate}.
     */
    public String getTemplateId()
    {
        return templateId;
    }

    /**
     * Get a constant identifier for a {@code ContentTemplate}, this is set by the content template
     * creator and is useful in the event that a person wants to use the same content template
     * across projects. One might think to use the {@link #getTemplateId()} however that changes for
     * every project, and is annoying to track across projects.
     *
     * @return The user-set identifier of the {@code ContentTemplate}.
     */
    public String getType()
    {
        return type;
    }

    /** Metadata used by surveys for submission.
     *
     * @return the metadata in the form of a {@code SurveyMetadata}
     */
    public JsonParser getMetadata() {
        return metadata;
    }

    /**
     * Set the time in epoch that indicates when this {@code Content} was created.
     *
     * @param createdAt The epoch time that indicates when this {@code Content} was created.
     */
    public void setCreatedAt(long createdAt) {
        this.createdAt = createdAt;
    }

    /**
     * Set the String representation of the data JSON object.
     *
     * @param dataAsJson The String representation of the data JSON object.
     */
    public void setDataAsJson(String dataAsJson) {
        this.dataAsJson = dataAsJson;
    }

    /**
     * Set the {@code description} {@link LocalizedValue} of the {@code Content}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code description}
     * attribute of this {@code {@link Content }}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param description The description of the {@code Content}.
     */
    public void setDescription(@NonNull String languageCode, @NonNull String description){
        this.descriptionObject.addValue(languageCode, description);
    }

    /**
     * Set the {@link LocalizedValue} of the description. The {@link LocalizedValue} object is used
     * in order to allow information about this {@code Content} to contain multiple languages and
     * therefore present the data in the language desired by the user of the application.
     *
     * @param value The {@link LocalizedValue} object that defines the description of the
     * {@code Content} in the various available languages.
     */
    public void setDescriptionObject(LocalizedValue value) {
        this.descriptionObject = value;
    }

    /**
     * Sets the favourite flag based {@code isFavourite} parameter. HOWEVER, this will not sync with
     * the server. Please use {@link #setFavourite(Context, BasicResultCallback)} for syncing with
     * the server. This method should only be used for internal purposes.
     *
     * @param isFavourite true to indicates locally (only) that this {@code Content} is a favourite,
     *                    false otherwise.
     */
    public void setIsFavourite(boolean isFavourite){
        this.isFavourite    = isFavourite;
    }

    /**
     * Set that this {@code Content} should be set as favourite.
     *
     * @param context The context of the activity that is setting this Content as a favourite.
     * @param callback {@code BasicResultCallback} to indicate whether or not saving a favourite was
     *                 successful.
     */
    public void setFavourite(Context context, BasicResultCallback callback){
        changeFavourite(context, true, callback);
    }

    /**
     * Set the url of the icon associated to this {@link Content} instance.
     *
     * @param icon The url of the icon associated to this {@link Content} instance.
     */
    public void setIcon(String icon) {
        this.icon = icon;
    }

    /**
     * Set the unique identifier for the {@code Content}. This identifier is used to delete the
     * {@code Content} as well as retrieve information that may be associated to it.
     *
     * @param id The unique GUID that represents this {@code Content}.
     */
    public void setId(@NonNull String id) {
        this.id = id;
    }

    /**
     * Set the labels that are associated to this instance of the {@code Content}.
     *
     * @param labels The list of labels that are associated to this piece of {@code Content}.
     */
    public void setLabels(@NonNull ArrayList<String> labels) {
        this.labels = labels;
    }

    /**
     * Set the time in epoch that indicates when this {@code Content} was last modified.
     *
     * @param modifiedAt The epoch time that indicates when this {@code Content} was last modified.
     */
    public void setModifiedAt(long modifiedAt) {
        this.modifiedAt = modifiedAt;
    }

    /**
     * Set the {@code name} {@link LocalizedValue} of the {@code Content}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code name} attribute of
     * this {@code Content}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param name The name of the {@code Content}.
     */
    public void setName(@NonNull String languageCode, @NonNull String name){
        this.nameObject.addValue(languageCode, name);
    }

    /**
     * Set the {@link LocalizedValue} of the name. The {@link LocalizedValue} object is used in
     * order to allow information about this {@code Content} to contain multiple languages and
     * therefore present the data in the language desired by the user of the application.
     *
     * @param value The {@link LocalizedValue} object that defines the bane of the {@code Content}
     *              in the various available languages.
     */
    public void setNameObject(LocalizedValue value) {
        this.nameObject = value;
    }

    /**
     * Set the template identifier of the {@code ContentTemplate} that was used to create this
     * {@link Content} instance.
     *
     * @param templateId The GUID of {@code ContentTemplate} that was used to create this
     * {@link Content} instance.
     */
    public void setTemplateId(String templateId) {
        this.templateId = templateId;
    }

    /**
     * Set a constant identifier for the {@code ContentTemplate} that created this {@link Content}
     * instance, this is set by the content template creator and is useful in the event that a
     * person wants to use the same content template across projects.
     *
     * @param type The user-set identifier of the {@code ContentTemplate}.
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * Unset that this {@code Content} should be a favourite.
     *
     * @param context The context of the activity that is unsetting this Content as a favourite.
     * @param callback {@code BasicResultCallback} to indicate whether or not saving a favourite was
     *                 successful.
     */
    public void unsetFavourite(Context context, BasicResultCallback callback) {
        changeFavourite(context, false, callback);
    }

     /** Set metadata
     *
     * @param metadata the metadata that implements the {@code JsonParser} class.
     */
    public void setMetadata(JsonParser metadata) {
        this.metadata = metadata;
    }

    /**
     * Get the data that this {@code Content} may contain.
     *
     * @param modelClass The template class that this data should deserialize into.
     * @param <T> The template class type
     * @return An instantiated object with the class of your template class, filled with the data
     * this {@code Content} has.
     * TODO: Missing @params xT
     * TODO: Missing @throws
     */
    public <T extends Parcelable> T getData(Context context, Class modelClass) throws FlybitsException {
        if (deserializedObject != null)
            return (T) deserializedObject;
        else
        {
            ContentDataResponse<T> response = new DeserializeContentDatum<T>(context, id, modelClass).fromJson(dataAsJson);

            if (response == null)
                throw new FlybitsException("Parsing Error");

            ArrayList<T> root = response.getItems();

            if (root != null && root.size() != 0)
                deserializedObject = (T) root.get(0);

            return (T) deserializedObject;
        }
    }

    //region Content Instance APIs
    /**
     * Get a list of {@code Content} based on the {@code ids} parameter which defines the
     * {@code Content} to be retrieved by ID.
     *
     * @param context The context of the application.
     * @param ids The unique identifiers of the Content to be retrieved.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * @return A {@code ContentResult} object for paging.
     */
    public static ListResult<Content> get(@NonNull final Context context, @NonNull final Collection<String> ids, final ListResultCallback<Content> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ListResult<Content> result = new ListResult<>(context, callback, executorService, handler);

        if (ids.size() == 0){
            result.setSuccess(new ArrayList<Content>());
            return result;
        }

        if (ids.size() > 50){
            throw new IllegalArgumentException("The size of your ids parameters must be less than 50. If you need to get more than 50 Content consider using multiple requests.");
        }

        executorService.execute(new Runnable() {
            public void run() {

                try{
                    StringBuilder url = new StringBuilder(API_CONTENT_INSTANCE + "?data=true&contentIds=");
                    url.append(TextUtils.join(",", ids));

                    DeserializeContentInstance singleDeserializer = new DeserializeContentInstance(context);
                    DeserializeListResponse<Content> deserializer = new DeserializeListResponse<>(singleDeserializer);
                    final Result<ArrayList<Content>> resultGetInstances = FlyAway.get(context, url.toString(),
                            null, (Map<String, String>) null, deserializer, "Content.getMultipleIds");

                    //Check for favourites
                    List<String> listOfStoredIds = CommonsDatabase.getDatabase(context).preferenceDAO().getIdsByKey(FAVOURITE_KEY);
                    if (listOfStoredIds.size() > 0) {
                        for (Content item : resultGetInstances.getResult()) {
                            item.setIsFavourite(listOfStoredIds.contains(item.getId()));
                        }
                    }

                    result.setResult(resultGetInstances);
                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });
        return result;
    }

    /**
     * Get a list of {@code Content} based on the {@link ContentParameters} parameter which defines
     * what options should be used to filter th e list of returned Content.
     *
     * @param context The context of the application.
     * @param params The {@link ContentParameters} that help filter the results of the query.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * @return A {@code ContentResult} object for paging.
     */
    public static ContentResult get(@NonNull final Context context, @NonNull final ContentParameters params, final PagedResultCallback<Content> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ContentResult result = new ContentResult(context, params, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    String url = params.getUrl();

                    DeserializeContentInstance singleDeserializer          = new DeserializeContentInstance(context);
                    DeserializePagedResponse<Content> deserializer          = new DeserializePagedResponse<>(singleDeserializer);
                    final Result<PagedResponse<Content>> resultGetInstances = FlyAway.get(context, url,
                            params, deserializer, "Content.get");


                    List<String> listOfIds = null;
                    if (resultGetInstances.getStatus() == RequestStatus.COMPLETED) {
                        if ((params.getQueryParams().get("offset") == null
                                || params.getQueryParams().get("offset").size() == 0
                                || params.getQueryParams().get("offset").get(0).equals("0"))
                                && (params.getCachingKey() != null)) {

                            listOfIds = CommonsDatabase.getDatabase(context).cachingEntryDAO().getIdsByCachingKey(params.getCachingKey());
                        }

                        int maxNumberOfRecordsToSave = (params.getCachingLimit() < resultGetInstances.getResult().getItems().size() + resultGetInstances.getResult().getPagination().getOffset())
                                ? params.getCachingLimit() : resultGetInstances.getResult().getItems().size();

                        //Check if Caching items are getting retrieved
                        if (params.getCachingKey() != null) {
                            ArrayList<CachingEntry> entries = new ArrayList<>();
                            ArrayList<Content> contentToCache = new ArrayList<>();
                            for (int i = 0; i < resultGetInstances.getResult().getItems().size(); i++) {
                                if (i >= maxNumberOfRecordsToSave){
                                    break;
                                }
                                Content content = resultGetInstances.getResult().getItems().get(i);
                                contentToCache.add(content);
                                entries.add(new CachingEntry(params.getCachingKey(), resultGetInstances.getResult().getItems().get(i).getId()));
                            }

                            if (listOfIds != null){
                                KernelDatabase.getDatabase(context).contentDao().insertAndDeleteInTransaction(listOfIds, contentToCache);
                                CommonsDatabase.getDatabase(context).cachingEntryDAO().deleteAllByCachingKey(params.getCachingKey());
                            }else {
                                KernelDatabase.getDatabase(context).contentDao().insert(contentToCache);
                            }
                            CommonsDatabase.getDatabase(context).cachingEntryDAO().insert(entries);
                        }

                        //Check for favourites
                        List<String> listOfStoredIds = CommonsDatabase.getDatabase(context).preferenceDAO().getIdsByKey(FAVOURITE_KEY);
                        if (listOfStoredIds.size() > 0) {
                            for (Content item : resultGetInstances.getResult().getItems()) {
                                item.setIsFavourite(listOfStoredIds.contains(item.getId()));
                            }
                        }
                    }
                    result.setResult(resultGetInstances, params);
                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });

        return result;
    }

    public static ListResult<Content> getFavourites(@NonNull final Context context, final ListResultCallback<Content> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ListResult<Content> result = new ListResult<>(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                List<String> listOfStoredIds = CommonsDatabase.getDatabase(context).preferenceDAO().getIdsByKey(FAVOURITE_KEY);

                //Remove duplicates
                Set<String> setOfContentIds = new HashSet<>(listOfStoredIds);
                get(context, setOfContentIds, callback);
            }
        });
        return result;
    }

    /**
     * Get a single {@code Content} by an id.
     *
     * @param context The context of the application.
     * @param id The id of the {@code Content} you would like to get.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * @return The {@code ObjectResult} that contains the network request.
     */
    public static ObjectResult<Content> getById(final Context context, final String id, final ObjectResultCallback<Content> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<Content> result = new ObjectResult<>(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    String url = String.format(API_CONTENT_INSTANCE_WITHDATA, id);
                    DeserializeContentInstance singleDeserializer          = new DeserializeContentInstance(context);
                    final Result<Content> resultGetInstance = FlyAway.get(context, url, singleDeserializer, "Content.getbyid", Content.class);

                    //Check for favourites
                    List<Preference> listOfStoredPreferences = CommonsDatabase.getDatabase(context).preferenceDAO().getIdsByKeyAndValue(FAVOURITE_KEY, id);
                    if (listOfStoredPreferences.size() > 0) {
                        resultGetInstance.getResult().setIsFavourite(true);
                    }

                    //Add to database or not?
                    result.setResult(resultGetInstance);
                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });

        return result;
    }

    /**
     * Get a list of {@code Content} based on the {@link ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param context The context of the application.
     * @param params The {@link ContentParameters} that help filter the results of the query.
     * @return A {@code ContentResult} object for paging.
     */
    public static ContentResult get(final Context context, final ContentParameters params){
       return get(context, params, null);
    }

    /**
     * Get a list of {@code Content} based on the {@link ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param context The context of the application.
     * @param params The {@link ContentParameters} that filters the reuslts of the data.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * @return A {@code ContentResult} object for paging.
     * TODO: Missing @throws
     * TODO: Missing @params
     */
    public static <T> ObjectResult<ContentDataResponse<T>> getData(final Context context, final ContentDataParameters params, final Class<T> classObj, final ObjectResultCallback<ContentDataResponse<T>> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<ContentDataResponse<T>> result = new ObjectResult<ContentDataResponse<T>>(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result<ContentDataResponse<T>> resultGetInstances = FlyContentData.get(context, params, classObj);
                    result.setResult(resultGetInstances);
                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });

        return result;
    }

    /**
     * Save data to a {@code Content} instance.
     *
     * @param context The context of the application.
     * @param objectToSave The data object to be saved.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * TODO: Missing @return
     */
    public BasicResult saveData(@NonNull final Context context, @NonNull final Object objectToSave, @NonNull final BasicResultCallback callback) {

        if (context == null){
            throw new IllegalArgumentException("The context parameter must not be null.");
        }

        if (objectToSave == null){
            throw new IllegalArgumentException("The object being saved must not be null.");
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    String json = ContentDataSerializer.serialize(objectToSave);
                    final Result resultPostData = FlyContentData.post(context, Content.this, json);
                    request.setResult(resultPostData);
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });

        return request;
    }

    //endregion

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Content> CREATOR = new Creator<Content>() {
        @Override
        public Content createFromParcel(Parcel in) {
            return new Content(in);
        }

        @Override
        public Content[] newArray(int size) {
            return new Content[size];
        }
    };

    @Override
    public String toString(){
        return String.format(Locale.getDefault(),"{id: %s, templateId: %s, nameObject: %s, descriptionObject: %s" +
                        ", createdAt: %d, modifiedAt: %d, icon: %s type: %s, isFavourite: %s}"
                ,id,templateId,nameObject,descriptionObject,createdAt,modifiedAt,icon,type,isFavourite);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this){
            return true;
        } else if (obj instanceof Content){

            //Initialize variables here so that we don't reference getters constantly
            Content otherContent = (Content)obj;
            String otherTemplateId = otherContent.getTemplateId();
            LocalizedValue otherNameObject = otherContent.getNameObject();
            LocalizedValue otherDescriptionObject = otherContent.getDescriptionObject();
            long otherCreatedAt = otherContent.getCreatedAt();
            long otherModifiedAt = otherContent.getModifiedAt();
            String otherIcon = otherContent.getIcon();
            String otherType = otherContent.getType();
            JsonParser otherMetaData = otherContent.getMetadata();
            List<String> labelsOther = otherContent.getLabels();

            //Each comparison is done on a separate line to make breakpoint debugging easy to visualize
            return otherContent.getId().equals(id)
                    && ( (otherTemplateId == null && templateId == null) || (otherTemplateId != null && otherTemplateId.equals(templateId) ) )
                    && ( (otherNameObject == null && nameObject == null) || ( otherNameObject != null && otherNameObject.equals(nameObject) ) )
                    && ( (otherDescriptionObject == null && descriptionObject == null) || ( otherDescriptionObject != null && otherDescriptionObject.equals(descriptionObject) ) )
                    && ( otherCreatedAt== createdAt )
                    && ( otherModifiedAt == modifiedAt )
                    && ( (otherIcon == null && icon == null) || ( otherIcon != null && otherIcon.equals(icon) ) )
                    && ( (otherType == null && type == null) || ( otherType != null && otherType.equals(type) ) )
                    //Labels are compared as a set so that order is not taken into account
                    && ( new HashSet<>(labelsOther).equals(new HashSet<>(getLabels())) )
                    && ( (otherMetaData == null && metadata == null) || ( otherMetaData != null && otherMetaData.toJson().equals(metadata.toJson()) ) );
        }else{
            return false;
        }
    }
}
