package com.flybits.android.kernel.db.caching;

import android.content.Context;
import android.support.annotation.NonNull;
import com.flybits.android.kernel.db.KernelDatabase;
import com.flybits.android.kernel.models.Group;
import com.flybits.commons.library.caching.FlybitsUIListObserver;

import java.util.List;

/**
 * The {@code ContentCacheObserver} class is responsible for creating an {@code Observer} for
 * {@link Group} retrieved from the Flybits server and stored on the local SQLite DB using ROOM.
 * This {@code Observer} listens for {@link Group} changes and notifies the UI whenever a change
 * is detected. It will also trigger the retrieval of the initial {@link Group} object from the DB
 * in order to initialize the view.
 */
public class GroupCacheObserver extends FlybitsUIListObserver<Group> {

    private static GroupCacheObserver INSTANCE = null;

    private GroupCacheObserver(final Context context){
        super(context);
    }

    @Override
    public void onLoad(List<String> cachingIds) {
        setItems(KernelDatabase.getDatabase(getContext()).groupDao().getLiveByCacheIds(cachingIds));
    }

    @Override
    public void add(@NonNull final String key, DataChanged<Group> callback){
        super.add(key, callback);
    }

    /**
     * As the {@code GroupCacheObserver} class acts as a {@code Singleton}, it is important to
     * initialize the {@code Singleton} with the proper attributes needed to function correctly.
     *
     * @param context The context of the application.
     * @return The instance of the {@code GroupCacheObserver} class which has either been
     * previously initialized or was initialized for the first time now.
     */
    public static GroupCacheObserver observe(Context context) {
        if (INSTANCE == null) {
            synchronized (GroupCacheObserver.class){
                if (INSTANCE == null){
                    INSTANCE = new GroupCacheObserver(context);
                }
            }
        }
        return INSTANCE;
    }

    @Override
    public void remove() {
        super.remove();
        if (getData() != null && !getData().hasActiveObservers()){
            INSTANCE = null;
        }
    }
}
