package com.flybits.android.kernel.utilities;

import com.flybits.android.kernel.models.LocalizedValue;
import com.flybits.android.kernel.models.PagedArray;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.util.ArrayList;

/**
 * The {@link ContentDataSerializer} class is a helper tool for converting a java POJO into
 * JSON.
 */
public class ContentDataSerializer {

    /**
     * Serializes a POJO into content.
     * @param object The JSON to read.
     * @return A serialized object.,
     * TODO: Missing @throws - Fix all comments here
     */
    public static String serialize(Object object) throws FlybitsException {
        return serializeObject(object).toString();
    }

    /**
     * Serializes an object and any sub-objects recursively.
     * @param currentObject The object to be serialized
     * @return JSON serialization of object
     * @throws FlybitsException If there was an issue with serialization
     */
    private static JSONObject serializeObject(Object currentObject) throws FlybitsException {

        JSONObject jsonObject = new JSONObject();

        for (Field f : currentObject.getClass().getDeclaredFields())
        {
            try {

                if (f.getName().equals("CREATOR"))
                    continue;

                //Null check, if localize value ignore otherwise set null
                if (f.get(currentObject) == null) {
                    if (!LocalizedValue.class.isAssignableFrom(f.getType()))
                        jsonObject.put(f.getName(), JSONObject.NULL);
                    continue;
                }

                //Paged Arrays
                if (PagedArray.class.isAssignableFrom(f.getType())) {
                    jsonObject.put(f.getName(), serializeArray(((PagedArray) f.get(currentObject)).getList()));
                }
                //Unpaged Arrays
                else if (ArrayList.class.isAssignableFrom(f.getType())) {
                    jsonObject.put(f.getName(), serializeArray((ArrayList) f.get(currentObject)));
                }
                //Localized Strings
                else if (LocalizedValue.class.isAssignableFrom(f.getType())) {

                    //Get or check if localization object exists
                    JSONObject localizationsObject = null;
                    try {
                        localizationsObject = jsonObject.getJSONObject("localizations");
                    } catch (JSONException e) {
                    }

                    //Create localizations object if none exists
                    if (localizationsObject == null) {
                        localizationsObject = new JSONObject();
                        jsonObject.put("localizations", localizationsObject);
                    }

                    LocalizedValue value = (LocalizedValue) f.get(currentObject);
                    for (String lang : value.getListOfSupportedLanguages()) {
                        if (!localizationsObject.has(lang))
                            localizationsObject.put(lang, new JSONObject());
                        localizationsObject.optJSONObject(lang).put(f.getName(), value.getValue(lang));
                    }
                }
                //Normal Primitives
                else if (checkIfPrimitiveType(f.getType())) {
                    jsonObject.put(f.getName(), f.get(currentObject));
                }
                //Objects
                else
                    jsonObject.put(f.getName(), serializeObject(f.get(currentObject)));
            } catch (IllegalAccessException e) {
                Logger.exception("ContentDataSerializer.serializeObject", e);
                throw new FlybitsException("ContentDataSerializer.serializeObject.RELFECTION_ERROR");
            } catch (JSONException e) {
                Logger.exception("ContentDataSerializer.serializeObject", e);
                throw new FlybitsException("ContentDataSerializer.serializeObject.JSON_SERIALIZATION_ERROR");
            }
        }

        return jsonObject;
    }

    /**
     * Serializes an array and any objects of primitives it contains
     * @param array The array to be serialized.
     * @return The JSON representation of this array.
     * @throws FlybitsException If there was an issue with serialization.
     */
    private static JSONArray serializeArray(ArrayList array) throws FlybitsException {

        JSONArray jsonArray = new JSONArray();

        for (int i = 0; i < array.size(); i++)
        {
            if (checkIfPrimitiveType(array.get(i).getClass()))
                jsonArray.put(array.get(i));
            else
                jsonArray.put(serializeObject(array.get(i)));
        }

        return jsonArray;
    }

    private static boolean checkIfPrimitiveType(Class type)
    {
        return  type.getName().equals("java.lang.String") ||
                type.getName().equals("java.lang.Boolean") || type.getName().equals("boolean") ||
                type.getName().equals("java.lang.Float") || type.getName().equals("float") ||
                type.getName().equals("java.lang.Double") || type.getName().equals("double") ||
                type.getName().equals("java.lang.Integer") || type.getName().equals("int") ||
                type.getName().equals("java.lang.Long") || type.getName().equals("long");
    }

}
