package com.flybits.android.kernel.api;

import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.android.kernel.deserializers.DeserializeContentDatum;
import com.flybits.android.kernel.models.Content;
import com.flybits.android.kernel.models.PagedArray;
import com.flybits.android.kernel.models.internal.ContentDataResponse;
import com.flybits.android.kernel.utilities.ContentDataParameters;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.HttpDefaultClass;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;

import java.io.IOException;

public class FlyContentData {

    static final String API_CONTENT_DATA        = "/kernel/content/instances/%1$s/data";
    static final String API_CONTENT_DATA_PAGED  = API_CONTENT_DATA + "?fields=%2$s&expand=%2$s&%2$s.offset=%3$d&%2$s.limit=%4$d";

    /**
     * Retrieves a paged portion of data from a content instance. This is the API endpoint of {@code PagedArray}.
     * @param context The context of the application.
     * @param array The {@code PagedArray} this request is being called for.
     * @param offset The paged offset to start from.
     * @param limit The number of list entries to request.
     * @param <T> The POJO class this PagedArray contains.
     * @return A {@code Result} object for setting callbacks and canceling the request.
     */
    public static <T> Result<ContentDataResponse<T>> getPaged(@NonNull Context context, @NonNull PagedArray array, long offset, long limit) throws FlybitsException {

        if (context == null){
            throw new IllegalArgumentException("The context parameter must not be null.");
        }

        if (array == null){
            throw new IllegalArgumentException("The array parameter must not be null.");
        }

        String url = String.format(FlybitsAPIConstants.constructGatewayURL(API_CONTENT_DATA_PAGED), array.getInstanceId(), array.getName(), offset, limit);

        try{
            Result result = new HttpDefaultClass.Builder(context, true, url)
                    .get()
                    .getResponse();

            Class clazz = array.getTemplateModel();
            if (result.getStatus() == RequestStatus.COMPLETED){
                ContentDataResponse response = new DeserializeContentDatum(context, array.getInstanceId(), clazz).fromJson(result.getResultAsString());
                if (response != null){
                    result.setResponse(response);
                }else{
                    result = new Result(new FlybitsException("Parsing Error"), result.getResultAsString());
                }
            }
            return result;
        }catch (IOException  e){
            Logger.exception("FlyContentData.get", e);
            throw new FlybitsException("Error Connecting to Flybits Server: FlyContentData_EXP1");
        }
    }

    /**
     * Perform a GET request to the Content Data API, given a set of params and a POJO to deserialize the response into.
     * @param context The context of the application.
     * @param params Any params that define how the response will be paged, or a unique ID to retrieve.
     * @param classObj The POJO to deserialize the data into.
     * @return A result object is returned so you can set the response callback.
     */
    public static <T> Result<ContentDataResponse<T>> get(@NonNull Context context, @NonNull ContentDataParameters params, @NonNull Class classObj) throws FlybitsException {

        if (context == null){
            throw new IllegalArgumentException("The context parameter must not be null.");
        }

        if (params == null || params.getInstanceID() == null){
            throw new IllegalArgumentException("The content object must not be null.");
        }

        if (classObj == null){
            throw new IllegalArgumentException("The class object must not be null.");
        }

        String url = String.format(FlybitsAPIConstants.constructGatewayURL(API_CONTENT_DATA), params.getInstanceID());
        try{
            Result result = new HttpDefaultClass.Builder(context, true, url)
                    .get()
                    .getResponse();

            if (result.getStatus() == RequestStatus.COMPLETED){
                ContentDataResponse response = new DeserializeContentDatum(context, params.getInstanceID(), classObj).fromJson(result.getResultAsString());
                if (response != null){
                    result.setResponse(response);
                }else{
                    result = new Result<ContentDataResponse>(new FlybitsException("Parsing Error"), result.getResultAsString());
                }
            }

            return result;
        }catch (IOException  e){
            Logger.exception("FlyContentData.get", e);
            throw new FlybitsException("Error Connecting to Flybits Server: FlyContentData_EXP2");
        }

    }

    /**
     * Perform a POST request to the Content Data API, to save any content data (IE: Survey).
     * @param context The context of the application.
     * @param content The {@link Content} this data belongs to.
     * @param json The data to save to the instance.
     * @return A result object is returned so you can set the response callback.
     */
    public static <T> Result<T> post(@NonNull Context context, @NonNull Content content, @NonNull String json) throws FlybitsException{

        if (context == null){
            throw new IllegalArgumentException("The context parameter must not be null.");
        }

        if (content == null || content.getId() == null){
            throw new IllegalArgumentException("The content object must not be null.");
        }

        if (json == null){
            throw new IllegalArgumentException("The json object must not be null.");
        }

        String url = String.format(FlybitsAPIConstants.constructGatewayURL(API_CONTENT_DATA), content.getId());
        try{

            Result<T> result = new HttpDefaultClass.Builder(context, true, url)
                    .post(json)
                    .getResponse();
            return result;
        }catch (IOException  e){
            Logger.exception("FlyContentData.post", e);
            throw new FlybitsException("Error Connecting to Flybits Server: FlyContentData_EXP3");
        }

    }

}
