package com.flybits.android.kernel.utilities;

import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link ContentDataParameters} class is used to define all the parameters set for the GET
 * request associated to {@link com.flybits.android.kernel.models.Content}.
 */
public class ContentDataParameters extends QueryParameters {

    private String instanceID;

    private ContentDataParameters(ContentDataParameters.Builder builder) {
        addPaging(builder.limit, builder.offset);
        this.instanceID = builder.instanceID;
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();
        return params;
    }

    public String getInstanceID(){
        return instanceID;
    }

    /**
     * {@code Builder} class used the create an {@link ContentDataParameters} class, which defines
     * all parameters used within the GET request for
     * {@link com.flybits.android.kernel.models.Content}.
     */
    public static class Builder {

        private long limit;
        private long offset;
        private String instanceID;

        /**
         * Default constructor to initializes all variables.
         */
        public Builder(){
            limit   = 0;
            offset  = -1;
        }

        /**
         * Adds a paging mechanism to the request based on the number of returned results wanted and
         * the offset of the next X number of results, where X is a limit.
         *
         * @param limit The maximum number of returned objects.
         * @param offset The offset of where the next X number of response objects will be returned
         *               from where X is the limit.
         *
         * @return A {@code Builder} object where additional {@code Options} filter settings can be
         * set.
         */
        public Builder addPaging(int limit, int offset){
            this.limit  = limit;
            this.offset = offset;
            return this;
        }

        public Builder addInstanceID(String id){
            this.instanceID = id;
            return this;
        }

        /**
         * Create a {@link ContentDataParameters} object that contains all parameters associated to
         * the GET request for {@link com.flybits.android.kernel.models.Experience}.
         *
         * @return {@link ContentDataParameters} which is used to create a valid
         * {@link com.flybits.android.kernel.models.Experience} GET request.
         */
        public ContentDataParameters build(){
            return new ContentDataParameters(this);
        }
    }
}
