package com.flybits.android.kernel.models;

import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.android.kernel.api.FlyContentData;
import com.flybits.android.kernel.db.KernelDatabase;
import com.flybits.android.kernel.deserializers.DeserializeContentDatum;
import com.flybits.android.kernel.deserializers.DeserializeContentInstance;
import com.flybits.android.kernel.models.internal.ContentDataResponse;
import com.flybits.android.kernel.models.results.ContentResult;
import com.flybits.android.kernel.utilities.ContentDataParameters;
import com.flybits.android.kernel.utilities.ContentParameters;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * An {@code Content} is a single instance of Content that is based off of an {@code ContentTemplate}. It can
 * have it's own name and description. Each {@code Content} has it's own {@code ContentData} that can be
 * requested.
 */
//TODO: Add comments on Setters/Getters
@Entity(tableName = "content")
public class Content implements Parcelable {

    public static final String API_CONTENT_RELEVANT_INSTANCES   = Experience.API+"/contents";
    public static final String API_CONTENT_INSTANCE             = "/kernel/content/instances";
    public static final String API_CONTENT_INSTANCE_WITHDATA    = API_CONTENT_INSTANCE+"/%s?data=true"; //TODO: Create a get that returns ALL instances (not just relevant)

    @PrimaryKey
    private String id;
    private String templateId;

    private LocalizedValue nameObject;
    private LocalizedValue descriptionObject;

    private long createdAt;
    private long modifiedAt;
    private String icon;
    private String dataAsJson;

    @Ignore
    private Parcelable deserializedObject;
    @Ignore
    private ArrayList<String> labels;

    /**
     * Default constructor used to allow ROOM to auto-initiate a {@code Content} object from Cache.
     */
    public Content(){}

    @Ignore
    private Content(@NonNull String id, @NonNull String templateId, String iconUrl, @NonNull String defaultLanguage,
                    @NonNull String defaultDeviceLangauge, long createdAt, long modifiedAt, ArrayList<String> labels){
        this.id             = id;
        this.templateId     = templateId;
        this.icon           = iconUrl;

        this.createdAt      = createdAt;
        this.modifiedAt     = modifiedAt;

        nameObject          = new LocalizedValue(defaultLanguage, defaultDeviceLangauge);
        descriptionObject   = new LocalizedValue(defaultLanguage, defaultDeviceLangauge);
        this.labels         = labels;
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code Content}.
     */
    @Ignore
    protected Content(Parcel in) {
        this(in.readString(), in.readString(), in.readString(), in.readString(), in.readString(),
                in.readLong(), in.readLong(), in.createStringArrayList());

        nameObject          = in.readParcelable(LocalizedValue.class.getClassLoader());
        descriptionObject   = in.readParcelable(LocalizedValue.class.getClassLoader());


        if (in.readInt() == 1){
            dataAsJson      = in.readString();
        }

        if (in.readInt() == 1){
            deserializedObject  = in.readParcelable(Parcelable.class.getClassLoader());
        }
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(id);
        dest.writeString(templateId);
        dest.writeString(icon);
        dest.writeString(nameObject.getEntityDefaultLanguage());
        dest.writeString(nameObject.getDeviceDefaultLanguage());

        dest.writeLong(createdAt);
        dest.writeLong(modifiedAt);

        dest.writeStringList(labels);

        dest.writeParcelable(nameObject, flags);
        dest.writeParcelable(descriptionObject, flags);

        dest.writeInt(dataAsJson != null ? 1 : 0);
        if (dataAsJson != null){
            dest.writeString(dataAsJson);
        }

        dest.writeInt(deserializedObject != null ? 1 : 0);
        if (deserializedObject != null){
            dest.writeParcelable(deserializedObject, flags);
        }
    }
    /**
     * Default constructor that initiates the basic attributes of an {@code Content}.
     *
     * @param id The unique identifier for the {@code Content}.
     * @param templateId The unique identifier representing the {@code ContentTemplate} this
     *                   instance is based off of.
     * @param iconUrl The url where the icon is stored.
     * @param defaultLanguage The default language that should be used to represent this
     *                        {@code Content}.
     * @param defaultDeviceLangauge The default language set by the device.
     * @param createdAt The time (in seconds - epoch) that indicates when this {@code Content}
     *                  was created.
     * @param modifiedAt The time (in seconds - epoch) that indicates when this {@code Content}
     *                  was last modified.
     * @param dataJson The JSON representation of data field for {@code Content}.
     * @param labels The list of labels associated to the {@code Content}.
     */
    public Content(@NonNull String id, @NonNull String templateId, String iconUrl, @NonNull String defaultLanguage,
                   @NonNull String defaultDeviceLangauge, long createdAt, long modifiedAt, @NonNull String dataJson,
                   ArrayList<String> labels) {

        this(id, templateId, iconUrl, defaultLanguage, defaultDeviceLangauge, createdAt, modifiedAt, labels);
        dataAsJson = dataJson;
    }

    /**
     * Gets the time (in seconds - epoch) that indicates when this {@code Content} was
     * created.
     * @return time in seconds.
     */
    public long getCreatedAt()
    {
        return createdAt;
    }

    /**
     * Get the description of the {@code Content} based on the selected default by the device. If the
     * device does not automatically save the language for whatever reason the default language from
     * the {@code #defaultLanguage} is used.
     *
     * @return The description of the {@code Content}.
     */
    public String getDescription()
    {
        return descriptionObject.getValue();
    }

    /**
     * Get the description of the {@code Content} based on the language given.
     *
     * @param langauge The language to get the description in.
     * @return The description of the {@code Content}.
     */
    public String getDescription(String langauge)
    {
        return descriptionObject.getValue(langauge);
    }

    /**
     * Get the {@link LocalizedValue} of the description of the {@code Content}.
     *
     * @return The description of the {@code Content} through the {@link LocalizedValue}.
     */
    public LocalizedValue getDescriptionObject(){
        return descriptionObject;
    }

    /**
     * Gets the icon url for the {@code Content}.
     *
     * @return The url where the icon for this {@code Content} is stored.
     */
    public String getIcon() { return icon; }

    /**
     * Gets the unique identifier representing this {@code Content}.
     *
     * @return The unique identifier of this {@code Content}.
     */
    public String getId()
    {
        return id;
    }

    /**
     * Get the data associated to this {@code Content} as a String variable.
     *
     * @return The String representation of the {@code Content}\'s data object.
     */
    public String getDataAsJson(){
        return dataAsJson;
    }

    /**
     * Get the list of labels that are associated to this piece of {@code Content}.
     *
     * @return The list of labels that are associated to this piece of {@code Content}.
     */
    public ArrayList<String> getLabels(){
        if (labels == null){
            labels  = new ArrayList<>();
        }
        return labels;
    }

    /**
     * Gets the time (in seconds - epoch) that indicates when this {@code Content} was
     * last modified.
     * @return time in seconds.
     */
    public long getModifiedAt()
    {
        return modifiedAt;
    }

    /**
     * Get the name of the {@code Content} based on the selected default by the device. If the
     * device does not automatically save the language for whatever reason the default language from
     * the {@code #defaultLanguage} is used.
     *
     * @return The name of the {@code Content}.
     */
    public String getName() { return nameObject.getValue(); }

    /**
     * Get the name of the {@code Content} based on the language given.
     *
     * @param language The language to get the name in.
     * @return The name of the {@code Content}.
     */
    public String getName(String language) { return nameObject.getValue(language); }

    /**
     * Get the {@link LocalizedValue} of the name of the {@code Content}.
     *
     * @return The name of the {@code Content} through the {@link LocalizedValue}.
     */
    public LocalizedValue getNameObject(){
        return nameObject;
    }

    /**
     * Gets the unique identifier of the {@code ContentTemplate} this instance is based off of.
     * @return The unique identifier of the {@code ContentTemplate}.
     */
    public String getTemplateId()
    {
        return templateId;
    }

    /**
     * Sets the {@code description} {@link LocalizedValue} of the {@code Content}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code description}
     * attribute of this {@code {@link Content }}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param description The description of the {@code Content}.
     */
    public void setDescription(@NonNull String languageCode, @NonNull String description){
        this.descriptionObject.addValue(languageCode, description);
    }

    /**
     * Sets the {@code name} {@link LocalizedValue} of the {@code Content}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code name} attribute of
     * this {@code Content}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param name The name of the {@code Content}.
     */
    public void setName(@NonNull String languageCode, @NonNull String name){
        this.nameObject.addValue(languageCode, name);
    }

    public void setId(String id) {
        this.id = id;
    }

    public void setTemplateId(String templateId) {
        this.templateId = templateId;
    }

    public void setNameObject(LocalizedValue name) {
        this.nameObject = name;
    }

    public void setDescriptionObject(LocalizedValue description) {
        this.descriptionObject = description;
    }

    public void setCreatedAt(long createdAt) {
        this.createdAt = createdAt;
    }

    public void setModifiedAt(long modifiedAt) {
        this.modifiedAt = modifiedAt;
    }

    public void setIcon(String icon) {
        this.icon = icon;
    }

    public void setDataAsJson(String dataAsJson) {
        this.dataAsJson = dataAsJson;
    }

    /**
     * Retrieves the data that this {@code Content} may contain.
     * @param modelClass The template class that this data should deserialize into.
     * @param <T> The template class type
     * @return An instantiated object with the class of your template class, filled with the data
     * this {@code Content} has.
     */
    public <T extends Parcelable> T getData(Context context, Class modelClass) throws FlybitsException {
        if (deserializedObject != null)
            return (T) deserializedObject;
        else
        {
            ContentDataResponse<T> response = new DeserializeContentDatum<T>(context, id, modelClass).fromJson(dataAsJson);

            if (response == null)
                throw new FlybitsException("Parsing Error");

            ArrayList<T> root = response.getItems();

            if (root != null && root.size() != 0)
                deserializedObject = (T) root.get(0);

            return (T) deserializedObject;
        }
    }

    //region Content Instance APIs

    /**
     * Asynchronously get a list of {@code Content} based on the
     * {@link com.flybits.android.kernel.utilities.ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param mContext The context of the application.
     * @param params The {@link ContentParameters} that help filter the results of the query.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * @return A {@code ContentResult} object for paging.
     */
    public static ContentResult get(final Context mContext, final ContentParameters params, final PagedResultCallback<Content> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ContentResult result = new ContentResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    String url = API_CONTENT_RELEVANT_INSTANCES;

                    if (params != null) {
                        if (params.getFilterType() == ContentParameters.Builder.ContentFilterType.ALL) {
                            url = API_CONTENT_INSTANCE +"/";
                        }else if (params.getFilterType() == ContentParameters.Builder.ContentFilterType.ID) {
                            url = API_CONTENT_INSTANCE + "/" + params.getFilterId();
                        }
                    }

                    DeserializeContentInstance singleDeserializer          = new DeserializeContentInstance(mContext);
                    DeserializePagedResponse<Content> deserializer          = new DeserializePagedResponse<Content>(singleDeserializer);
                    final Result<PagedResponse<Content>> resultGetInstances = FlyAway.get(mContext, url,
                            params, deserializer, "Content.get");

                    if (resultGetInstances.getStatus() == RequestStatus.COMPLETED) {
                        if (params == null
                                || params.getQueryParams().get("offset") == null
                                || params.getQueryParams().get("offset").size() == 0
                                || params.getQueryParams().get("offset").get(0).equals("0")) {

                            KernelDatabase.getDatabase(mContext).contentDao().clear();
                        }

                        KernelDatabase.getDatabase(mContext).contentDao().
                                insertOrReplaceNewContent(resultGetInstances.getResult().getItems());
                    }

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            result.setResult(resultGetInstances, params);
                        }
                    });

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });
                }
            }
        });

        return result;
    }

    /**
     * Asynchronously get a list of {@code Content} based on the
     * {@link com.flybits.android.kernel.utilities.ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param mContext The context of the application.
     * @param params The {@link ContentParameters} that help filter the results of the query.
     * @return A {@code ContentResult} object for paging.
     */
    public static ContentResult get(final Context mContext, final ContentParameters params){
       return get(mContext, params, null);
    }

    /**
     * Asynchronously get a list of {@code Content} based on the
     * {@link com.flybits.android.kernel.utilities.ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param mContext The context of the application.
     *
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * @return A {@code ContentResult} object for paging.
     */
    public static <T> ObjectResult<ContentDataResponse<T>> getData(final Context mContext, final ContentDataParameters params, final Class<T> classObj, final ObjectResultCallback<ContentDataResponse<T>> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<ContentDataResponse<T>> result = new ObjectResult<ContentDataResponse<T>>(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result<ContentDataResponse<T>> resultGetInstances = FlyContentData.get(mContext, params, classObj);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            result.setResult(resultGetInstances);
                        }
                    });
                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            result.setFailed(e);
                        }
                    });
                }
            }
        });

        return result;
    }
    //endregion

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Content> CREATOR = new Creator<Content>() {
        @Override
        public Content createFromParcel(Parcel in) {
            return new Content(in);
        }

        @Override
        public Content[] newArray(int size) {
            return new Content[size];
        }
    };
}
