package com.flybits.android.kernel.utilities;

import com.flybits.android.kernel.models.Group;
import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link GroupParameters} class is used to define all the parameters set for the GET
 * request associated to {@link Group}.
 */
public class GroupParameters extends QueryParameters {

    private GroupParameters(GroupParameters.Builder builder) {
        addPaging(builder.limit, builder.offset);
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();
        return params;
    }

    /**
     * {@code Builder} class used the create an {@link GroupParameters} class, which defines
     * all parameters used within the GET request for
     * {@link Group}.
     */
    public static class Builder {

        private long limit;
        private long offset;

        /**
         * Default constructor to initializes all variables.
         */
        public Builder(){
            limit   = 0;
            offset  = -1;
        }

        /**
         * Adds a paging mechanism to the request based on the number of returned results wanted and
         * the offset of the next X number of results, where X is a limit.
         *
         * @param limit The maximum number of returned objects.
         * @param offset The offset of where the next X number of response objects will be returned
         *               from where X is the limit.
         *
         * @return A {@code Builder} object where additional {@code Options} filter settings can be
         * set.
         */
        public Builder addPaging(int limit, int offset){
            this.limit  = limit;
            this.offset = offset;
            return this;
        }

        /**
         * Create a {@link GroupParameters} object that contains all parameters associated to
         * the GET request for {@link Group}.
         *
         * @return {@link GroupParameters} which is used to create a valid
         * {@link Group} GET request.
         */
        public GroupParameters build(){
            return new GroupParameters(this);
        }
    }
}
