package com.flybits.android.kernel.utilities;

import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link ContentParameters} class is used to define all the parameters set for the GET
 * request associated to {@link com.flybits.android.kernel.models.Content}.
 */
public class ContentParameters  extends QueryParameters {

    private Builder.ContentFilterType filterType;
    private String filterId = null;

    private ContentParameters(ContentParameters.Builder builder) {
        addPaging(builder.limit, builder.offset);
        filterType = builder.filterType;
        filterId = builder.filterId;
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();

        if (filterType == ContentParameters.Builder.ContentFilterType.ALL || filterType == ContentParameters.Builder.ContentFilterType.ID) {
            ArrayList<String> limitParam = new ArrayList<>();
            limitParam.add( String.valueOf(true));
            params.put("data", limitParam);
        }

        return params;
    }

    public Builder.ContentFilterType getFilterType()
    {
        return filterType;
    }

    public String getFilterId()
    {
        return filterId;
    }

    /**
     * {@code Builder} class used the create an {@link ContentParameters} class, which defines
     * all parameters used within the GET request for
     * {@link com.flybits.android.kernel.models.Content}.
     */
    public static class Builder {

        public enum ContentFilterType {ALL, RELEVANT, ID};
        private ContentFilterType filterType;
        private String filterId = null;
        private long limit;
        private long offset;

        /**
         * Default constructor to initializes all variables.
         */
        public Builder(){
            limit   = 0;
            offset  = -1;
        }

        /**
         * Adds a paging mechanism to the request based on the number of returned results wanted and
         * the offset of the next X number of results, where X is a limit.
         *
         * @param limit The maximum number of returned objects.
         * @param offset The offset of where the next X number of response objects will be returned
         *               from where X is the limit.
         *
         * @return A {@code Builder} object where additional {@code Options} filter settings can be
         * set.
         */
        public Builder addPaging(int limit, int offset){
            this.limit  = limit;
            this.offset = offset;
            return this;
        }

        public Builder addFilter(ContentFilterType filter, Object... params) throws FlybitsException
        {
            filterType = filter;

            if (filterType == ContentFilterType.ID)
            {
                if (params.length >= 1 && params[0] instanceof String)
                    filterId = (String) params[0];
                else
                    throw new FlybitsException("ContentParam Error: invalid params");

            }
            return this;
        }

        /**
         * Create a {@link ContentParameters} object that contains all parameters associated to
         * the GET request for {@link com.flybits.android.kernel.models.Experience}.
         *
         * @return {@link ContentParameters} which is used to create a valid
         * {@link com.flybits.android.kernel.models.Experience} GET request.
         */
        public ContentParameters build(){

            return new ContentParameters(this);
        }
    }
}
