package com.flybits.android.kernel.models;

import android.arch.persistence.room.Entity;
import android.arch.persistence.room.PrimaryKey;
import android.arch.persistence.room.TypeConverters;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.android.kernel.KernelScope;
import com.flybits.android.kernel.deserializers.DeserializeExperience;
import com.flybits.android.kernel.models.results.ExperienceResult;
import com.flybits.android.kernel.db.converters.LocalizedValueConverters;
import com.flybits.android.kernel.utilities.ExperienceParameters;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * An {@code Experience} is indicates when a specific piece of {@code Content} should be available
 * to an end user. The "when" in this case is initiated by {@code Rules} that are triggered based on
 * changes to a user's context which can be obtained from device sensors, and/or 3rd party APIs.
 */

@Entity(tableName = "experiences")
public class Experience implements Parcelable{

    public static final String API      = KernelScope.ROOT + "/experiences";

    @PrimaryKey
    private String id;
    private String creatorID;
    private boolean isActive;
    private long createdAt;
    private long modifiedAt;
    private LocalizedValue name;

    @TypeConverters({LocalizedValueConverters.class})
    private ArrayList<String> labels;
    @TypeConverters({LocalizedValueConverters.class})
    private LocalizedValue description;

    private Experience(String creatorID, boolean isActive){
        this.creatorID          = creatorID;
        this.isActive           = isActive;
    }

    /**
     * Constructor that initializes the basic attributes of an {@code Experience}. This constructor
     * should be used when constructing a new {@code Experience}. Otherwise if you are retrieving
     * previously created {@code Experiences} then the
     * {@link #Experience(String, String, String, boolean, long, long, String, ArrayList)} should be
     * used.
     *
     * @param context The context of the application.
     * @param defaultLanguage The default language that should be used to represent this
     *                        {@code Experience}.
     * @param isActive Indicates whether {@code Rules} associated to this {@code Experience} will be
     *                 processed for {@code Context} delivery.
     */
    public Experience(@NonNull Context context, @NonNull String defaultLanguage, boolean isActive){

        this(SharedElements.getUserID(context), isActive);

        ArrayList<String> deviceDefaultLanguages    = SharedElements.getEnabledLanguagesAsArray(context);
        String deviceDefaultLanguage    = (deviceDefaultLanguages.size() == 0)
                ? defaultLanguage : deviceDefaultLanguages.get(0);

        this.name               = new LocalizedValue(defaultLanguage, deviceDefaultLanguage);
        this.description        = new LocalizedValue(defaultLanguage, deviceDefaultLanguage);
    }

    /**
     * Constructor that initializes the basic attributes of an {@code Experience}. This constructor
     * is will provide a thorough definition of an {@code Experience}. If you are creating a new
     * {@code Experience}, then you should use the {@link #Experience(Context, String, boolean)}
     * constructor.
     *
     * @param id The unique identifier for the {@code Experience}.
     * @param creatorID The unique identifier representing the user that created this
     *                  {@code Experience}.
     * @param defaultLanguage The default language that should be used to represent this
     *                        {@code Experience}.
     * @param deviceDefaultLanguage The default language set by the device.
     * @param isActive Indicates whether {@code Rules} associated to this {@code Experience} will be
     *                 processed for {@code Context} delivery.
     * @param createdAt The time (in seconds - epoch) that indicates when this {@code Experience}
     *                  was created.
     * @param modifiedAt The time (in seconds - epoch) that indicates when this {@code Experience}
     *                   was last modified.
     */
    public Experience(@NonNull String id, @NonNull String creatorID, @NonNull String defaultLanguage,
                      boolean isActive, long createdAt, long modifiedAt, @NonNull String deviceDefaultLanguage,
                      ArrayList<String> labels) {
        this(creatorID, isActive);
        this.id                 = id;
        this.createdAt          = createdAt;
        this.modifiedAt         = modifiedAt;

        this.name               = new LocalizedValue(defaultLanguage, deviceDefaultLanguage);
        this.description        = new LocalizedValue(defaultLanguage, deviceDefaultLanguage);
        this.labels             = labels;
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code Content}.
     */
    protected Experience(Parcel in) {
        this (in.readString(), in.readInt() == 1);
        id = in.readString();
        createdAt = in.readLong();
        modifiedAt = in.readLong();
        labels = in.createStringArrayList();
        name = in.readParcelable(LocalizedValue.class.getClassLoader());
        description = in.readParcelable(LocalizedValue.class.getClassLoader());
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(creatorID);
        dest.writeInt(isActive ? 1 : 0);
        dest.writeString(id);
        dest.writeLong(createdAt);
        dest.writeLong(modifiedAt);
        dest.writeStringList(labels);
        dest.writeParcelable(name, flags);
        dest.writeParcelable(description, flags);
    }

    /**
     * Create this {@code Experience} based on the attributes defined within this object.
     *
     * @param mContext The context of the application.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code ObjectResult} that contains the network request for creating
     * {@code Experiences}.
     */
    public ObjectResult<Experience> create(final Context mContext,
                                           final ObjectResultCallback<Experience> callback) throws FlybitsException{

        if (getNameObject().getListOfSupportedLanguages().size() == 0){
            throw new FlybitsException("You must have at least one name defined");
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<Experience> request = new ObjectResult<Experience>(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    DeserializeExperience deserializer = new DeserializeExperience(mContext);
                    String json             = deserializer.toJson(Experience.this);
                    final Result<Experience> created = FlyAway.post(mContext, API, json, deserializer, "Experience.create", Experience.class);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(created);
                        }
                    });

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * Delete this {@code Experience} based on the identifier set through the
     * {@link #Experience(String, String, String, boolean, long, long, String, ArrayList)}
     * constructor.
     *
     * @param mContext The context of the application.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code BasicResult} that contains the network request for deleting a specific
     * {@code Experience}.
     */
    public BasicResult delete(final Context mContext, final BasicResultCallback callback) throws FlybitsException{

        if (id == null){
            throw new FlybitsException("You must have an id set to able to delete an Experience");
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result deleted = FlyAway.delete(mContext, API, "Experience.delete", id);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(deleted);
                        }
                    });

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }

    /**
     * An asynchronous request to get a list of {@code Experiences} based on the
     * {@link com.flybits.android.kernel.utilities.ExperienceParameters} parameter which defines
     * which options should be used to filter the list of returned parameters.
     *
     * @param mContext The context of the application.
     * @param params The {@link ExperienceParameters} that indicate filters that should be placed on
     *               the request.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     */
    public static ExperienceResult get(final Context mContext, final ExperienceParameters params,
                                       final PagedResultCallback<Experience> callback){


        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ExperienceResult result = new ExperienceResult(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    DeserializeExperience singleDeserializaer            = new DeserializeExperience(mContext);
                    DeserializePagedResponse<Experience> deserializer    = new DeserializePagedResponse<Experience>(singleDeserializaer);
                    final Result<PagedResponse<Experience>> getExperiences = FlyAway.get(mContext,
                            API, params, deserializer, "FlyExperience.get");

                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            result.setResult(getExperiences, params);
                        }
                    });
                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });
                }
            }
        });
        return result;
    }


    /**
     * Gets the time (in milliseconds - epoch) that indicates when this {@code Experience} was
     * created.
     * @return time in seconds.
     */
    public long getCreatedAt() {
        return createdAt;
    }

    /**
     * Gets the unique identifier representing the user that created this {@code Experience}.
     * @return The user identifier of the person who created this {@code Experience}.
     */
    public String getCreatorID() {
        return creatorID;
    }

    /**
     * Get the description of the {@code Experience} based on the selected default by the device. If
     * the device does not automatically save the language for whatever reason the default language
     * from the {@code defaultLanguage} is used.
     *
     * @return The description of the {@code Experience}.
     */
    public String getDescription(){
        return description.getValue();
    }

    /**
     * Get the description of the {@code Experience} based on the {@code languageCode} parameter. If
     * the {@code languageCode} parameter is null or a {@code languageCode} is currently not present
     * in the {@link LocalizedValue} description value then null is returned.
     *
     * @return The description of the {@code Experience} based on the {@code languageCode}, if no
     * match is found then null is returned.
     */
    public String getDescription(String languageCode){
        return description.getValue(languageCode);
    }

    /**
     * Get the {@link LocalizedValue} of the description of the {@code Experience}.
     *
     * @return The description of the {@code Experience} through the {@link LocalizedValue}.
     */
    public LocalizedValue getDescriptionObject(){
        return description;
    }

    /**
     * Gets the unique identifier representing this {@code Experience}.
     * @return The unique identifier of this {@code Experience}.
     */
    public String getId() {
        return id;
    }

    /**
     * Get the list of labels associated to the {@code Experience}.
     *
     * @return The list of labels associated to the {@code Experience}.
     */
    public ArrayList<String> getLabels(){

        if (labels == null){
            labels = new ArrayList<>();
        }
        return labels;
    }

    /**
     * Gets the time (in milliseconds - epoch) that indicates when this {@code Experience} was last
     * modified.
     * @return time in seconds.
     */
    public long getModifiedAt() {
        return modifiedAt;
    }

    /**
     * Get the name of the {@code Experience} based on the selected default by the device. If the
     * device does not automatically save the language for whatever reason the default language from
     * the {@code defaultLanguage} is used.
     *
     * @return The name of the {@code Experience}.
     */
    public String getName(){
        return name.getValue();
    }

    /**
     * Get the name of the {@code Experience} based on the {@code languageCode} parameter. If
     * the {@code languageCode} parameter is null or a {@code languageCode} is currently not present
     * in the {@link LocalizedValue} name value then null is returned.
     *
     * @return The name of the {@code Experience} based on the {@code languageCode}, if no match is
     * found then null is returned.
     */
    public String getName(String languageCode){
        return name.getValue(languageCode);
    }

    /**
     * Get the {@link LocalizedValue} of the name of the {@code Experience}.
     *
     * @return The name of the {@code Experience} through the {@link LocalizedValue}.
     */
    public LocalizedValue getNameObject(){
        return name;
    }

    /**
     * Indicates whether {@code Rules} associated to this {@code Experience} will be processed for
     * {@code Context} delivery.
     * @return true if the {@code Experience} is currently available, false otherwise.
     */
    public boolean isActive() {
        return isActive;
    }

    /**
     * Sets the {@code description} {@link LocalizedValue} of the {@code Experience}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code description}
     * attribute of this {@code Experience}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param description The description of the {@code Experience}.
     */
    public void setDescription(@NonNull String languageCode, @NonNull String description){
        this.description.addValue(languageCode, description);
    }

    /**
     * Sets the {@code name} {@link LocalizedValue} of the {@code Experience}. The
     * {@link LocalizedValue} contains all the supported languages for the {@code name} attribute of
     * this {@code Experience}.
     *
     * @param languageCode The Language code that should be used as the key for the String
     *                     representation of the name.
     * @param name The name of the {@code Experience}.
     */
    public void setName(@NonNull String languageCode, @NonNull String name){
        this.name.addValue(languageCode, name);
    }

    /**
     * Update this {@code Experience} based on all the newly set attributes.
     *
     * @param mContext The context of the application.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code ObjectResult} that contains the network request for updating a specific
     * {@code Experience}.
     */
    public ObjectResult<Experience> update(final Context mContext,
                                          ObjectResultCallback<Experience> callback) throws FlybitsException{

        if (id == null){
            throw new FlybitsException("You must have an id set to able to delete an Experience");
        }

        if (getNameObject().getListOfSupportedLanguages().size() == 0){
            throw new FlybitsException("You must have at least one name defined");
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<Experience> request = new ObjectResult<Experience>(mContext, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    DeserializeExperience deserializer  = new DeserializeExperience(mContext);
                    String json                         = deserializer.toJson(Experience.this);
                    final Result<Experience> updated    = FlyAway.put(mContext, API, json, deserializer, "Experience.update", Experience.class);
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setResult(updated);
                        }
                    });

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            request.setFailed(e);
                        }
                    });
                }
            }
        });
        return request;
    }


    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Experience> CREATOR = new Creator<Experience>() {
        @Override
        public Experience createFromParcel(Parcel in) {
            return new Experience(in);
        }

        @Override
        public Experience[] newArray(int size) {
            return new Experience[size];
        }
    };
}
