package com.flybits.android.kernel.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.android.kernel.api.FlyContentData;
import com.flybits.android.kernel.models.internal.ContentDataResponse;
import com.flybits.android.kernel.utilities.ContentDataDeserializer;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * A object representing an array of data in a Content Data response. Used for arrays when defining
 * your POJO. After initial load, more data can be requested using {@code getNext}.
 * @param <T> The class type this array holds.
 */
public class PagedArray<T extends Parcelable> implements Parcelable{

    private String mContentInstanceId;
    private String mFieldPath;
    private long mCurrentOffset = 0;
    private long mTotalRecords;
    private long mLimit;
    private ArrayList<T> mCurrentDownloaded;
    private Class mTemplateModel;

    public PagedArray(String contentInstanceId, String fieldPath, Class model,
                      long totalRecords, long limit, long offset, ArrayList<T> list) {
        mContentInstanceId      = contentInstanceId;
        mFieldPath              = fieldPath;
        mTotalRecords           = totalRecords;
        mLimit                  = limit;
        mCurrentOffset          = offset + limit;
        if (mCurrentOffset >= mTotalRecords) {
            mCurrentOffset = mTotalRecords;
        }
        mCurrentDownloaded      = list;
        mTemplateModel          = model;
    }

    public PagedArray(Parcel in) {
        mContentInstanceId      = in.readString();
        mFieldPath              = in.readString();
        mCurrentOffset          = in.readLong();
        mTotalRecords           = in.readLong();
        mLimit                  = in.readLong();

        int listSize            = in.readInt();
        if (listSize > 0){
            Class<?> type       = (Class<?>) in.readSerializable();
            mCurrentDownloaded  = new ArrayList<>(listSize);
            in.readList(mCurrentDownloaded, type.getClassLoader());
        }

        try {
            mTemplateModel      = Class.forName(in.readString());
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * Requests the next page of data from the server.
     * @param limit The max amount of items to download for the next page.
     */
    public void getNext(Context context, int limit) throws FlybitsException {
        if (!hasNext())
        return;

        Result<ContentDataResponse<T>> result = FlyContentData.getPaged(context, PagedArray.this, mCurrentOffset, limit);
        PagedArray<T> newPagedArray = ContentDataDeserializer.extractPagedArray(result.getResult().getItems().get(0), mFieldPath);

        mTotalRecords = newPagedArray.mTotalRecords;

        mCurrentOffset += limit;
        mLimit = newPagedArray.mLimit;
        mCurrentDownloaded.addAll(newPagedArray.getList());

        if (mCurrentOffset >= mTotalRecords)
            mCurrentOffset = mTotalRecords;
    }

    /**
     * Asynchronously get a list of {@code Content} based on the
     * {@link com.flybits.android.kernel.utilities.ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param context The context of the application.
     * @param limit The max amount of items to download for the next page.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * {@code Experiences} based on the {@code params}.
     */
    public void getNext(final Context context, final long limit, final BasicResultCallback callback){

        if (!hasNext()){
            callback.onSuccess();
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult result = new BasicResult(context, callback, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result<ContentDataResponse<T>> pageResult = FlyContentData.getPaged(context, PagedArray.this, mCurrentOffset, limit);
                    PagedArray<T> newPagedArray = ContentDataDeserializer.extractPagedArray(pageResult.getResult().getItems().get(0), mFieldPath);
                    if (newPagedArray == null){
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                result.setFailed(pageResult.getException());
                            }
                        });
                    }else{
                        mTotalRecords = newPagedArray.mTotalRecords;

                        mCurrentOffset += limit;
                        mLimit = newPagedArray.mLimit;
                        mCurrentDownloaded.addAll(newPagedArray.getList());

                        if (mCurrentOffset >= mTotalRecords)
                            mCurrentOffset = mTotalRecords;

                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                result.setResult(pageResult);
                            }
                        });
                    }

                }catch (final FlybitsException e){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            result.setFailed(e);
                        }
                    });
                }
            }
        });
    }

    /**
     * Returns if there is more items to request from the server.
     * @return True if there is more items from the server.
     */
    public boolean hasNext()
    {
        return mCurrentOffset < mTotalRecords;
    }

    /**
     * Gets a list of currently downloaded items.
     */
    //TODO: Make sure mCurrentDownloaded is not null if null return empty set
    public ArrayList<T> getList()
    {
        return mCurrentDownloaded;
    }

    /**
     * Gets the instance id of the {@code Content} that held this PagedArray.
     * @return The instance id of the parent {@code Content}.
     */
    public String getInstanceId()
    {
        return mContentInstanceId;
    }

    /**
     * Gets the field path to this PagedArray.
     * @return The field path to this PagedArray.
     */
    public String getPath()
    {
        return mFieldPath;
    }

    /**
     * Gets the name of this PagedArray.
     * @return The name of this PagedArray.
     */
    public String getName()
    {
        if (mFieldPath.contains("."))
            return mFieldPath.substring(mFieldPath.lastIndexOf(".") + 1);
        return mFieldPath;
    }

    /**
     * Get's the model of the template that held this Paged Array. Used for paging and deserialization.
     * @return A class object of the template this PagedArray resides in.
     */
    public Class getTemplateModel() {
        return mTemplateModel;
    }


    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mContentInstanceId);
        dest.writeString(mFieldPath);
        dest.writeLong(mCurrentOffset);
        dest.writeLong(mTotalRecords);
        dest.writeLong(mLimit);
        dest.writeInt(mCurrentDownloaded.size());

        if (mCurrentDownloaded != null && mCurrentDownloaded.size() > 0){
            final Class<?> objectsType = mCurrentDownloaded.get(0).getClass();
            dest.writeSerializable(objectsType);
            dest.writeList(mCurrentDownloaded);
        }

        dest.writeString(mTemplateModel.getName());
    }

    public static final Creator<PagedArray> CREATOR = new Creator<PagedArray>() {
        @Override
        public PagedArray createFromParcel(Parcel in) {
            return new PagedArray(in);
        }

        @Override
        public PagedArray[] newArray(int size) {
            return new PagedArray[size];
        }
    };
}
