package com.flybits.android.kernel.models;

import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

//TODO: Consider moving to the Commons SDK as other SDKs may want to use it

/**
 * The {@code LocalizedValue} class is responsible for storing various attributes about an entity,
 * such as the {@link Experience} that can be translated. For example, within the Flybits ecosystem
 * we can have the name of the experience change based on the preferred language of an application
 * user. There are many examples of attributes that can be localized, each one of these is outlined
 * within each Entity.
 */
public class LocalizedValue implements Parcelable{

    private HashMap<String, String> localizedValue;
    private String defaultLanguage;
    private String deviceDefaultLanguage;

    /**
     * Constructor that is used the define the default attributes needed in case the user attempts
     * to access a localized value with a language that is not set.
     *
     * @param entityDefaultLanguage The default language of the entity such as {@link Experience}.
     * @param deviceDefaultLanguage The default language of the device. This can be obtained from
     *                              the device settings or overridden by the application based on
     *                              its own internal logic.
     * @throws InvalidParameterException If either the {@code entityDefaultLanguage} or
     * {@code deviceDefaultLanguage} attributes are not in the following format "en" (Must be
     * exactly 2 characters long).
     */
    public LocalizedValue(@NonNull String entityDefaultLanguage, @NonNull String deviceDefaultLanguage)
            throws InvalidParameterException{

        checkLanguageConditions(deviceDefaultLanguage);
        checkLanguageConditions(entityDefaultLanguage);

        localizedValue          = new HashMap<>();
        this.defaultLanguage    = entityDefaultLanguage;
        this.deviceDefaultLanguage  = deviceDefaultLanguage;
    }

    public LocalizedValue(Parcel in) {
        this (in.readString(), in.readString());
        int hashMapSize         = in.readInt();
        for (int i = 0; i < hashMapSize; i++) {
            localizedValue.put(in.readString(), in.readString());
        }
    }

    /**
     * Adds a {@code value} for a specific {@code language}.
     *
     * @param language The language that should be added for this specific entity. This attribute
     *                 must be submitted in the following format "en".
     * @param value The value for the language, such as setting "en" for the {@link Experience}
     *              name.
     * @throws InvalidParameterException If the {@code language} attribute is not in the following
     * format "en" (Must be exactly 2 characters long).
     */
    public void addValue(@NonNull String language, @NonNull String value) throws InvalidParameterException{
        checkLanguageConditions(language);
        localizedValue.put(language, value);
    }

    /**
     * Get the default language for the device or the manually overridden value defined within the
     * application.
     * @return THe default language of the device in the following format - "en".
     */
    public String getDeviceDefaultLanguage(){
        return deviceDefaultLanguage;
    }

    /**
     * Get the default language for this entity such as an {@link Experience} entity.
     * @return THe default language of the entity in the following format - "en".
     */
    public String getEntityDefaultLanguage(){
        return defaultLanguage;
    }

    /**
     * Get the list of supported languages (as 2-letter codes) for this {@code LocalizedValue}.
     *
     * @return The list of supported language codes.
     */
    public ArrayList<String> getListOfSupportedLanguages(){
        return new ArrayList<>(localizedValue.keySet());
    }

    /**
     * Get the default value of the entity based on the language that is set. The order of
     * processing will check the {@link #getDeviceDefaultLanguage()} method first to see if it is
     * set. If it is not set, then {@link #getEntityDefaultLanguage()} will be used to obtain the
     * default value of this entity's localized attribute.
     *
     * @return The string representation of the value based on the default languages set.
     */
    public String getValue(){

        if (localizedValue.containsKey(deviceDefaultLanguage)){
            return localizedValue.get(deviceDefaultLanguage);
        }

        return localizedValue.get(defaultLanguage);
    }

    /**
     * Get the value of the entity based on the {@code language} parameter.
     *
     * @return The string representation of the value based on the default languages set.
     * @throws InvalidParameterException If the {@code language} attribute is not in the following
     * format - "en" (Must be exactly 2 characters long).
     */
    public String getValue(String language){

        if (language != null && localizedValue.containsKey(language)) {
            return localizedValue.get(language);
        }

        return null;
    }

    /**
     * Remove the attribute that is associated to this entity. This is useful in the event that you
     * previously stored a value under the language "fr" and now you want to remove it.
     * @param language The language that you would like to remove from the list of saved entities.
     */
    public void removeProperty(@NonNull String language){
        localizedValue.remove(language);
    }

    private void checkLanguageConditions(String language) throws InvalidParameterException{
        if (language == null || language.length() != 2){
            throw new InvalidParameterException("You language must be 2 character long such as \"en\"");
        }
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(defaultLanguage);
        dest.writeString(deviceDefaultLanguage);
        dest.writeInt(localizedValue != null ? localizedValue.size() : 0);
        for (Map.Entry<String, String> entry : localizedValue.entrySet()) {
            dest.writeString(entry.getKey());
            dest.writeString(entry.getValue());
        }
    }

    public static final Creator<LocalizedValue> CREATOR = new Creator<LocalizedValue>() {
        @Override
        public LocalizedValue createFromParcel(Parcel in) {
            return new LocalizedValue(in);
        }

        @Override
        public LocalizedValue[] newArray(int size) {
            return new LocalizedValue[size];
        }
    };
}
