package com.flybits.android.kernel.utilities;

import android.support.annotation.NonNull;

import com.flybits.android.kernel.models.Content;
import com.flybits.commons.library.models.internal.QueryBuilder;
import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link ContentParameters} class is used to define all the parameters set for the GET
 * request associated to {@link com.flybits.android.kernel.models.Content}.
 */
public class ContentParameters  extends QueryParameters {

    private String locId;
    private double latitude;
    private double longitude;
    private double radius;
    private String url;
    private String templateId;

    private ContentParameters(ContentParameters.Builder builder) {
        super(builder);

        this.locId          = builder.locId;

        if (locId != null) {
            this.latitude   = builder.latitude;
            this.longitude  = builder.longitude;
            this.radius     = builder.radius;
            if (builder.templateId != null){
                templateId  = builder.templateId;
            }

            url             = Content.API_CONTENT_INSTANCE;
        }else{
            url             = Content.API_CONTENT_RELEVANT_INSTANCES;
        }
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();
        if (locId != null){

            ArrayList<String> latitudeParam = new ArrayList<>();
            latitudeParam.add( String.valueOf(latitude));
            params.put("latitude", latitudeParam);

            ArrayList<String> longitudeParam = new ArrayList<>();
            longitudeParam.add( String.valueOf(longitude));
            params.put("longitude", longitudeParam);

            ArrayList<String> radiusParam = new ArrayList<>();
            radiusParam.add( String.valueOf(radius));
            params.put("radius", radiusParam);

            ArrayList<String> locParam = new ArrayList<>();
            locParam.add( String.valueOf(locId));
            params.put("loc", locParam);

            ArrayList<String> dataParam = new ArrayList<>();
            dataParam.add( String.valueOf(true));
            params.put("data", dataParam);

            if (templateId != null) {
                ArrayList<String> templateIdParam = new ArrayList<>();
                templateIdParam.add(String.valueOf(templateId));
                params.put("templateId", templateIdParam);
            }
        }
        return params;
    }

    /**
     * Returns the URL of the API that should be used to retrieve {@link Content}.
     *
     * @return The URL of the API that should be used to retrieve {@link Content}.
     */
    public String getUrl() {
        return url;
    }

    /**
     * {@code Builder} class used the create an {@link ContentParameters} class, which defines
     * all parameters used within the GET request for
     * {@link com.flybits.android.kernel.models.Content}.
     */
    public static class Builder extends QueryBuilder<Builder>{

        private String locId;
        private double latitude;
        private double longitude;
        private double radius;
        private String templateId;

        /**
         * Default constructor to initializes all variables.
         */
        public Builder(){
            super();
        }

        /**
         * Sets sorting based on location. Therefore, the content will be sorted based on the
         * location data type.
         *
         * @param loc  location field to search on (required).
         * @param latitude latitude to search from (required).
         * @param longitude longitude to search from (required).
         * @param radius radius to search in.
         */
        public Builder setSorting(@NonNull String loc, double latitude, double longitude, long radius) {
            return setSorting(loc, latitude, longitude, radius, null);
        }

        /**
         * Sets sorting based on location. Therefore, the content will be sorted based on the
         * location data type.
         *
         * @param loc  location field to search on (required).
         * @param latitude latitude to search from (required).
         * @param longitude longitude to search from (required).
         * @param radius radius to search in.
         * @param templateId The unique identifier of the Content Template to perform the search
         *                   for.
         */
        public Builder setSorting(@NonNull String loc, double latitude, double longitude,
                                  long radius, String templateId) {
            this.locId      = loc;
            this.latitude   = latitude;
            this.longitude  = longitude;
            this.radius     = radius;
            this.templateId = templateId;
            return this;
        }

        /**
         * Create a {@link ContentParameters} object that contains all parameters associated to
         * the GET request for {@link com.flybits.android.kernel.models.Experience}.
         *
         * @return {@link ContentParameters} which is used to create a valid
         * {@link com.flybits.android.kernel.models.Experience} GET request.
         */
        public ContentParameters build(){

            return new ContentParameters(this);
        }
    }
}
