package com.flybits.android.kernel.models;

import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.android.kernel.utilities.ContentParameters;

/**
 * The {@link LocationData} class is used to store location data about {@link Content}. As part of
 * the {@link LocationData} a distance is included that represents the distance between
 * {@link Content} and a specific location.
 */
public class LocationData implements Parcelable {

    private double lat;
    private double lng;
    private double distance;
    private Unit unit;

    /**
     * Default Constructor used to set the latitude, longitude, distance and unit type of a
     * location.
     *
     * @param lat The latitude of the location.
     * @param lng The longitude of the location.
     * @param distance The distance between the given geo-point and the {@link Content} location.
     * @param unit The String representation of the unit that the {@code distance} is calculated in.
     */
    public LocationData(double lat, double lng, double distance, String unit){
        this.lat    = lat;
        this.lng    = lng;
        this.distance   = distance;
        this.unit   = Unit.fromValue(unit);
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code Content}.
     */
    public LocationData(Parcel in) {
        lat = in.readDouble();
        lng = in.readDouble();
        distance = in.readDouble();
        unit    = Unit.fromValue(in.readString());
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeDouble(lat);
        dest.writeDouble(lng);
        dest.writeDouble(distance);
        dest.writeString(unit.getKey());
    }

    public static final Creator<LocationData> CREATOR = new Creator<LocationData>() {
        @Override
        public LocationData createFromParcel(Parcel in) {
            return new LocationData(in);
        }

        @Override
        public LocationData[] newArray(int size) {
            return new LocationData[size];
        }
    };

    /**
     * Gets the latitude of the location.
     *
     * @return the latitude of the location.
     */
    public double getLat() {
        return lat;
    }

    /**
     * Gets the longitude of the location.
     *
     * @return the longitude of the location.
     */
    public double getLng() {
        return lng;
    }

    /**
     * Gets the distance between the given geo-point and the {@link Content} location.
     *
     * @return the distance between the given geo-point and the {@link Content} location.
     */
    public double getDistance() {
        return distance;
    }

    /**
     * Gets the {@link Unit} that the {@link #getDistance()} is calculated in.
     *
     * @return The {@link Unit} that the {@link #getDistance()} is calculated in.
     */
    public Unit getUnit() {
        return unit;
    }

    /**
     * The {@link Unit} is an enumerator that represents all of {@link Unit}s that
     * {@link #getDistance()} can be calculated in.
     */
    public enum Unit{
        /**
         * Indicates the unit measured in Meters.
         */
        METERS("meter"){
            public double toFeet(double distance){return distance*3.28084;}
            public double toKilometers(double distance){return distance*0.001;}
            public double toMiles(double distance){return distance*0.000621371;}
            public double toMeters(double distance){return distance;}

        },

        /**
         * Indicates the unit measured in Feet.
         */
        FEET("feet"){
            public double toMeters(double distance){return distance*0.3048000097536;}
            public double toKilometers(double distance){return distance*0.0003048;}
            public double toMiles(double distance){return distance*0.000189394;}
            public double toFeet(double distance){return distance;}
        },

        /**
         * Indicates the unit measured in Kilometers.
         */
        KILOMETERS("kilometer"){
            public double toMeters(double distance){return distance*1000;}
            public double toFeet(double distance){return distance*3280.84;}
            public double toMiles(double distance){return distance*0.6213712;}
            public double toKilometers(double distance){return distance;}
        },

        /**
         * Indicates the unit measured in Miles.
         */
        MILES("miles"){
            public double toMeters(double distance){return distance*1609.34405;}
            public double toFeet(double distance){return distance*5280;}
            public double toKilometers(double distance){return distance*1.61;}
            public double toMiles(double distance){return distance;}
        };

        private final String key;

        Unit(String value){
            this.key  = value;
        }

        /**
         * Get the key of the {@link Unit} that indicates which type of measurement unit was used to
         * calculate the {@link LocationData#getDistance()}.
         *
         * @return The String representation of the {@link Unit}.
         */
        public String getKey() {
            return this.key;
        }

        /**
         * Converts the {@code value} into a valid {@link Unit}. If an invalid entry is provided
         * then the default {@link Unit#METERS} will be used.
         *
         * @param value The String representation of the {@link Unit}.
         * @return The {@link Unit} which represents the {@code value} parameter.
         */
        public static Unit fromValue(String value){
            if (value == null){
                return METERS;
            }

            for(Unit type : Unit.values()) {
                if(type.getKey().equalsIgnoreCase(value)) {
                    return type;
                }
            }
            return METERS;
        }

        /**
         * Calculates the distance into Imperical `feet` measurement unit.
         *
         * @param distance A value to calculate distance in feet.
         * @return The new distance in feet.
         */
        public abstract double toFeet(double distance);

        /**
         * Calculates the distance into Metric `Kilometers` measurement unit.
         *
         * @param distance A value to calculate distance in kilometers.
         * @return The new distance in kilometers.
         */
        public abstract double toKilometers(double distance);

        /**
         * Calculates the distance into Metric `Meters` measurement unit.
         *
         * @param distance A value to calculate distance in meters.
         * @return The new distance in meters.
         */
        public abstract double toMeters(double distance);

        /**
         * Calculates the distance into Metric `Miles` measurement unit.
         *
         * @param distance A value to calculate distance in miles.
         * @return The new distance in miles.
         */
        public abstract double toMiles(double distance);
    }
}
