package com.flybits.context.plugins.oauth;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * The {@code Provider} class is used to define where the {@code OAuth} information he being
 * obtained from. Provider are 3rd-party services which Google, Facebook, Twitter, etc. that a user
 * can log into and then the Flybits system can obtained either their profile information or some
 * other data which can be converted to Contextual information.
 */
public class Provider implements Parcelable {

    private String id;
    private String name;
    private String displayName;

    /**
     * Constructor used for un-flattening a {@code Provider} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Provider} parcel.
     */
    public Provider(Parcel in){
        id             = in.readString();
        name           = in.readString();
        displayName    = in.readString();
    }

    /**
     * Constructor used to define a {@code Provider} using preset information about the name and
     * unique identifier used to define the {@code Provider}.
     *
     * @param id The unique identifier for this {@code Provider}.
     * @param name The name of the {@code Provider} that is used internally by the system.
     * @param displayName The text-friendly name of the {@code Provider} so that the end-user can
     *                    easily understand which {@code Provider} they are connecting to.
     */
    public Provider(String id, String name, String displayName) {
        this.id             = id;
        this.name           = name;
        this.displayName    = displayName;
    }

    /**
     * The text-friendly name of the {@code Provider} so that the end-user can easily understand
     * which {@code Provider} they are connecting to.
     *
     * @return The text-friendly name of the {@code Provider}.
     */
    public String getDisplayName() {
        return displayName;
    }

    /**
     * The unique identifier of the {@code Provider}.
     *
     * @return The unique String identifier of the {@code Provider}.
     */
    public String getId() {
        return id;
    }

    /**
     * The name of the {@code Provider}.
     *
     * @return The name of the {@code Provider}.
     */
    public String getName() {
        return name;
    }


    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(id);
        out.writeString(name);
        out.writeString(displayName);
    }

    /**
     * Parcelable.Creator that instantiates {@code Provider} objects
     */
    public static final Creator<Provider> CREATOR = new Creator<Provider>(){
        public Provider createFromParcel(Parcel in){
            return new Provider(in);
        }

        public Provider[] newArray(int size){
            return new Provider[size];
        }
    };

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Provider provider = (Provider) o;

        return id != null ? id.equals(provider.id) : provider.id == null;
    }
}
