package com.flybits.concierge.viewproviders

import android.content.Context
import android.graphics.Color
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.WebView
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.commons.library.utils.Utilities
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.R
import com.flybits.concierge.UserActionHandler
import com.flybits.concierge.activities.SimpleCardDetailActivity
import com.flybits.concierge.models.SimpleCard
import com.flybits.concierge.viewactionhandlers.SimpleCardUserActionHandler
import com.flybits.concierge.viewholders.FlybitsViewHolder

/**
 * ViewProvider for Type 1 StreamLine Content.
 */
class SimpleCardViewProvider internal constructor(
    private val contentAnalytics: ContentAnalytics?
    , private val userActionHandler: UserActionHandler<SimpleCard>
) : FlybitsViewProvider<SimpleCard> {

    /**
     * Constructor used for instantiating the [SimpleCardViewProvider] with the default [UserActionHandler] for [SimpleCard].
     *
     * @param context The application's [Context].
     */
    constructor(context: Context) : this(ContentAnalytics(context), SimpleCardUserActionHandler())

    /**
     * Constructor used for instantiating the [SimpleCardViewProvider] with a specific [SimpleCardViewProvider].
     *
     * @param context The application's [Context].
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     * When overriding only a subset of the default handling logic, extend and override [SimpleCardUserActionHandler].
     */
    constructor(context: Context, userActionHandler: UserActionHandler<SimpleCard>)
            : this(ContentAnalytics(context), userActionHandler)


    override fun getClassType(): Class<SimpleCard> {
        return SimpleCard::class.java
    }

    override fun getContentType(): String {
        return CONTENT_TYPE
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<SimpleCard> {
        val view = LayoutInflater.from(parent.context)
            .inflate(R.layout.flybits_streamline_links, parent, false)
        return SimpleCardViewHolder(view, contentAnalytics, userActionHandler)
    }


    companion object {

        const val CONTENT_TYPE = "concierge-card-link"
        const val componentID_link = "link"
        const val componentType = "link"

        private fun configureView(
            data: SimpleCard,
            simpleCardView: View,
            userActionHandler: UserActionHandler<SimpleCard>,
            flybitsNavigator: FlybitsNavigator,
            contentAnalytics: ContentAnalytics?
        ) {
            val bannerView: ImageView =
                simpleCardView.findViewById(R.id.flybits_streamline_link_imgHeader)
            val headerView: TextView =
                simpleCardView.findViewById(R.id.flybits_streamline_link_header)
            val titleView: TextView =
                simpleCardView.findViewById(R.id.flybits_streamline_link_title)
            val descriptionView: WebView =
                simpleCardView.findViewById(R.id.flybits_streamline_link_description)
            val linkViewProvider: TextView =
                simpleCardView.findViewById(R.id.flybits_streamline_link_action)

            data.image?.value.let {
                if (it.isNullOrEmpty()) {
                    bannerView.visibility = View.GONE
                } else {
                    Glide.with(bannerView.context).load(it)
                        .apply(RequestOptions().centerCrop()).into(bannerView)
                }
            }

            data.header?.value.let {
                if (it.isNullOrEmpty()) {
                    headerView.visibility = View.GONE
                } else {
                    headerView.text = it
                }
            }

            titleView.text = data.title?.value

            data.description?.value.let {
                when {
                    it.isNullOrEmpty() -> {
                        descriptionView.visibility = View.GONE
                    }
                    else -> {
                        val htmlData = Utilities.toRTF(it)
                        descriptionView.loadData(htmlData, "text/html", "UTF-8")
                        descriptionView.setBackgroundColor(Color.TRANSPARENT)
                        descriptionView.isVerticalScrollBarEnabled = false
                    }
                }
            }

            linkViewProvider.text = data.link?.text?.value.toString()
            linkViewProvider.setOnClickListener {
                userActionHandler.onUserAction(
                    SimpleCardUserActionHandler.LINK_CLICKED,
                    data,
                    flybitsNavigator
                )

                // analytics expected format
                // ctx.flybits.contentAnalytics.query.componentEngaged.{Content ID}.{Component ID}.{Component Name}.{Component Type}
                val componentName = linkViewProvider.text.toString().replace(".", "")
                val key =
                    data.content.id + "." + componentID_link + "." + componentName + "." + componentType
                contentAnalytics?.trackComponentEngaged(key)
                contentAnalytics?.trackEngaged(data.content)
            }
        }
    }

    class SimpleCardViewHolder(
        val view: View,
        val contentAnalytics: ContentAnalytics?,
        private val userActionHandler: UserActionHandler<SimpleCard>
    ) : FlybitsViewHolder<SimpleCard>(view) {
        private val single: View = view.findViewById(R.id.flybits_streamline_link_single)
        private val emptyView: View = view.findViewById(R.id.smartrewards_streamline_link_emptyTxt)

        override fun bindData(data: SimpleCard, flybitsNavigator: FlybitsNavigator) {
            single.visibility = View.GONE
            emptyView.visibility = View.GONE

            when (data.title?.value) {
                null -> {
                    emptyView.visibility = View.VISIBLE
                    single.visibility = View.GONE
                }
                else -> {
                    emptyView.visibility = View.GONE
                    single.visibility = View.VISIBLE
                    configureView(
                        data,
                        view,
                        userActionHandler,
                        flybitsNavigator,
                        contentAnalytics
                    )
                }
            }

        }
    }

    override fun onPushOpened(model: SimpleCard, flybitsNavigator: FlybitsNavigator) {
        model.let {
            val bundle = Bundle()
            bundle.putParcelable(SimpleCardDetailActivity.EXTRA_SIMPLECARD_DETAILS, it)
            bundle.putParcelable(
                SimpleCardDetailActivity.EXTRA_USER_ACTION_HANDLER,
                userActionHandler
            )
            flybitsNavigator.openActivity(SimpleCardDetailActivity::class.java, bundle)
        }
    }
}