package com.flybits.concierge.repository.push

import android.arch.lifecycle.LiveData
import android.arch.lifecycle.MutableLiveData
import android.content.Context
import com.flybits.android.kernel.db.KernelDatabase
import com.flybits.android.kernel.models.Content
import com.flybits.android.push.models.newPush.ContentPush
import com.flybits.android.push.models.newPush.Push
import com.flybits.android.push.utils.PushQueryParameters
import com.flybits.android.push.utils.SortOrder
import com.flybits.commons.library.api.results.callbacks.ListResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Pagination
import com.flybits.concierge.FlybitsViewProviderGetter
import com.flybits.concierge.models.BaseTemplate
import com.flybits.concierge.models.NotificationContent
import com.flybits.concierge.repository.ModelConverter
import com.flybits.concierge.repository.RepositoryResponse
import com.flybits.concierge.repository.content.ContentGetter
import java.util.*
import java.util.concurrent.Executors

/**
 * This class is responsible for dealing with [Push] and [Content]
 * in scenarios where [Content] is embedded within a [Push], creating [NotificationContent].
 *
 * @param pushGetter Class that provides push notifications.
 * @param contentGetter Class that provides content.
 * @param modelConverter Class that converts Content to BaseTemplate.
 */
class NotificationContentRepository(
    private val pushGetter: PushGetter,
    private val contentGetter: ContentGetter,
    private val modelConverter: ModelConverter,
    private val context: Context
) {

    companion object {
        private const val JSON_KEY_URL = "url"
        private const val JSON_KEY_VALUE = "value"
        private const val JSON_KEY_CONTENT = "contentId"

        /**
         * Create new instance of [NotificationContentRepository].
         *
         * @param context
         * @param flybitsViewProviderGetter
         *
         * @return new instance of [NotificationContentRepository].
         */
        fun create(
            context: Context,
            flybitsViewProviderGetter: FlybitsViewProviderGetter
        ): NotificationContentRepository {
            val contentGetter = ContentGetter(context)
            val pushGetter = PushGetter(context)
            val modelConverter = ModelConverter(contentGetter, flybitsViewProviderGetter)
            return NotificationContentRepository(pushGetter, contentGetter, modelConverter, context)
        }
    }

    /**
     * Retrieve all content tied to notifications received on this device. Maximum 20 content
     * instances will be returned. Duplicate content instances are removed.
     *
     * @param callback callback used for providing data response
     *
     * @return live data with the response
     */
    fun getNotificationContentRemote(
        callback: ObjectResultCallback<List<NotificationContent>>? = null
    )
            : LiveData<RepositoryResponse<List<NotificationContent>>> {
        val liveData = MutableLiveData<RepositoryResponse<List<NotificationContent>>>()

        val params = PushQueryParameters.Builder()
            .setSortOrder(SortOrder.DESCENDING)
            .setPaging(20, 0)
            .build()

        pushGetter.getPush(
            params = params,
            pagedResultCallback = object : PagedResultCallback<Push> {

                override fun onSuccess(items: ArrayList<Push>, pagination: Pagination) {
                    Logger.d("NotificationContentRepository: got items=$items")
                    val contentIds = items.asSequence().map {
                        if (it is ContentPush) it.contentId else null
                    }.filterNotNull().toList().distinct()

                    val itemsContentList = items.asSequence().map {
                        if (it is ContentPush) it else null
                    }.distinctBy { it?.contentId }.filterNotNull().toList().distinct()

                    if (contentIds.isEmpty()) {
                        val response = listOf<NotificationContent>()
                        liveData.value = RepositoryResponse(data = response)
                        callback?.onSuccess(response)
                        return
                    }

                    contentGetter.getContent(contentIds, object : ListResultCallback<Content> {
                        override fun onSuccess(items: ArrayList<Content>) {
                            Logger.d("NotificationContentRepository: got content=$items")

                            val notificationContent: List<NotificationContent> =
                                contentIds.mapIndexed { index, contentId ->
                                    //Ensure order is according to push endpoint response
                                    val content = items.find { it.id == contentId }
                                    if (content != null) {

                                        // Update the push request Id for associated Content
                                        Executors.newSingleThreadExecutor().execute {
                                            content.pushRequestId = itemsContentList[index].parentId
                                            KernelDatabase.getDatabase(context).contentDao().update(
                                                content
                                            )
                                        }

                                        val baseTemplate =
                                            modelConverter.contentToBaseTemplate(content)
                                        Logger.d("NotificationContentRepository: base template converted=$baseTemplate")
                                        baseTemplate?.let { baseTemplateNonNull ->
                                            NotificationContent(baseTemplateNonNull, 0)
                                        }
                                    } else null
                                }.filterNotNull()

                            liveData.value = RepositoryResponse(data = notificationContent)
                            callback?.onSuccess(notificationContent)
                        }

                        override fun onException(exception: FlybitsException) {
                            Logger.d("NotificationContentRepository: exception=$exception")
                            liveData.value = RepositoryResponse(error = exception)
                            callback?.onException(exception)
                        }

                    })
                }

                override fun onLoadedAllItems() {
                }

                override fun onException(exception: FlybitsException) {
                    Logger.d("NotificationContentRepository: exception=${exception.message}")
                    liveData.value = RepositoryResponse(error = exception)
                    callback?.onException(exception)
                }

            })

        return liveData
    }

    /**
     * Retrieve [BaseTemplate] for a particular content id.
     * This method can be used in a scenario where a user receives a push notification
     * and the associated content needs to be displayed in the concierge.
     *
     * @param contentId contentId of the [BaseTemplate]
     * @param callback Callback used for notifying the caller about the results. Null by default,
     * the alternative is to use the [LiveData] returned by this method.
     *
     * @return [LiveData] that will receive the [RepositoryResponse]
     */
    fun getBaseTemplate(
        contentId: String,
        callback: ObjectResultCallback<BaseTemplate>? = null
    ): LiveData<RepositoryResponse<BaseTemplate>> {
        val liveData = MutableLiveData<RepositoryResponse<BaseTemplate>>()

        //Continue to retrieve content using the id
        contentGetter.getContent(contentId, object : ObjectResultCallback<Content> {
            override fun onSuccess(item: Content) {
                val baseTemplate = modelConverter.contentToBaseTemplate(item)
                if (baseTemplate != null) {
                    liveData.value = RepositoryResponse(baseTemplate, null)
                    callback?.onSuccess(baseTemplate)
                } else {
                    val exception = FlybitsException("Unable to covert content to base template")
                    liveData.value = RepositoryResponse(null, exception)
                    callback?.onException(exception)
                }
            }

            override fun onException(e: FlybitsException) {
                liveData.value = RepositoryResponse(null, e)
                callback?.onException(e)
            }
        })
        return liveData
    }
}