package com.flybits.concierge.guidedoptin.presentation.view

import android.arch.lifecycle.ViewModelProviders
import android.os.Bundle
import android.support.v4.app.Fragment
import android.support.v4.view.ViewPager.OnPageChangeListener
import android.text.Html
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import android.widget.TextView
import com.afollestad.materialdialogs.MaterialDialog
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.*
import com.flybits.concierge.enums.DisplayType
import com.flybits.concierge.guidedoptin.TAG_BENEFITS
import com.flybits.concierge.guidedoptin.UI_TRANSPARENT
import com.flybits.concierge.guidedoptin.presentation.adapter.ViewPagerAdapter
import com.flybits.concierge.guidedoptin.presentation.viewmodel.GuidedOptInViewModel
import com.flybits.concierge.utils.setCustomTextColorStyle
import com.flybits.concierge.utils.setTextViewHTML
import kotlinx.android.synthetic.main.flybits_fragment_guided_benefits.*

/**
 * This BenefitsOptInFragment is responsible for Benefits Guided Opt In UI.
 */

internal class BenefitsOptInFragment : Fragment(), OptedStateChangeListener{
    private val viewModel: GuidedOptInViewModel by lazy {
        ViewModelProviders.of(activity!!).get(GuidedOptInViewModel::class.java)
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        return inflater.inflate(R.layout.flybits_fragment_guided_benefits, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val concierge = FlybitsConcierge.with(context)

        val supportActionBarTitle: TextView? =
            activity?.findViewById(R.id.flybits_guided_toolbar_title)
        supportActionBarTitle?.text = getString(R.string.flybits_guided_benefits_appbar_title)

        flybits_guided_benefits_next.isEnabled = false
        flybits_guided_benefits_checkbox.visibility = View.VISIBLE
        flybits_guided_benefits_checkbox.setOnCheckedChangeListener { _, isChecked ->
            flybits_guided_benefits_next.isEnabled = isChecked
        }

        // Set click on next.
        flybits_guided_benefits_next.setOnClickListener {
            uiOnButtonClick()
            if (!InternalPreferences.is2PhaseOptIn(context)) {
                concierge.optIn(object : BasicResultCallback {
                    override fun onException(exception: FlybitsException) {
                        showDialog()
                        uiOnButtonReset()
                    }

                    override fun onSuccess() {
                        uiOnButtonReset()
                        viewModel.onBenefitsContinueClicked()
                    }
                })
            } else {
                concierge.optIn2PhaseCallBack.onSuccess()
            }
        }

        // Check if end user is opted in to flybits.
        concierge.isOptedInLocal(object : ObjectResultCallback<Boolean> {
            /**
             * Indicates that the request failed.
             *
             * @param exception The [FlybitsException] that indicates the exception that was thrown.
             */
            override fun onException(exception: FlybitsException) {
                Logger.appendTag(TAG_BENEFITS).d("User has opted in successfully.")
            }

            /**
             * Indicates that the request was successfully made.
             *
             * @param item The item that is expected to be returned.
             */
            override fun onSuccess(item: Boolean) {
                // If already opt in , hide disclaimer text and checkbox.
                if (item) {
                    flybits_guided_benefits_checkbox.visibility = View.GONE
                    flybits_guided_benefits_disclaimer.visibility = View.GONE
                    flybits_guided_benefits_next.isEnabled = true
                }
            }
        })



        viewModel.getBenefitsOptIn()?.let {

            // Check if the mandatoryDisclaimerCheckBox is set to false.
            if (!it.disclaimerCheckBox) {
                flybits_guided_benefits_next.isEnabled = true
                flybits_guided_benefits_checkbox.visibility = View.INVISIBLE
                flybits_guided_benefits_checkbox.width = 0
            }

            //Set Values
            flybits_guided_benefits_next.text = getString(it.buttonTitle)
            flybits_guided_benefits_title.text = getString(it.title)
            context?.let { context ->
                setTextViewHTML(
                    flybits_guided_benefits_disclaimer,
                    getString(it.disclaimer),
                    context
                )
            } ?: run {
                flybits_guided_benefits_disclaimer.text =
                    getString(R.string.flybits_guided_disclaimer_error)
            }

            if (it.content.isNotEmpty()) {
                // Set Adapter for ViewPager and Dot Indicator
                addIndicators(0, it.content.size)
                val sliderPagerAdapter =
                    ViewPagerAdapter(activity, it.content)
                flybits_guided_benefits_viewPager?.apply {
                    adapter = sliderPagerAdapter
                    offscreenPageLimit = it.content.size //Add size
                    measure(
                        LinearLayout.LayoutParams.MATCH_PARENT,
                        LinearLayout.LayoutParams.WRAP_CONTENT
                    )
                    addOnPageChangeListener(object : OnPageChangeListener {
                        override fun onPageScrolled(
                            position: Int,
                            positionOffset: Float,
                            positionOffsetPixels: Int
                        ) {
                        }

                        override fun onPageSelected(position: Int) {
                            addIndicators(position, it.content.size)
                        }

                        override fun onPageScrollStateChanged(state: Int) {
                        }
                    })
                }
            }
        }
    }

    companion object {
        @JvmStatic
        fun newInstance() =
            BenefitsOptInFragment()
    }

    private fun addIndicators(currentPage: Int, benefitsItemSize: Int) {
        if (benefitsItemSize > 1) {
            val dotsIndicator: Array<TextView?> = arrayOfNulls(benefitsItemSize)
            flybits_guided_benefits_viewPager_indicator!!.removeAllViews()
            for (i in dotsIndicator.indices) {
                dotsIndicator[i] = TextView(activity?.applicationContext)
                if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
                    dotsIndicator[i]!!.text = Html.fromHtml("&#8226;", Html.FROM_HTML_MODE_COMPACT)
                } else {
                    dotsIndicator[i]!!.text = Html.fromHtml("&#8226;")
                }
                dotsIndicator[i]!!.textSize = 35F
                dotsIndicator[i]!!.setTextColor(
                    setCustomTextColorStyle(
                        context!!,
                        R.style.Concierge_BenefitsOptIn_ViewPager_PageIndicatorInActive
                    )
                )
                flybits_guided_benefits_viewPager_indicator!!.addView(dotsIndicator[i])
            }
            if (dotsIndicator.isNotEmpty()) dotsIndicator[currentPage]?.setTextColor(
                setCustomTextColorStyle(
                    context!!,
                    R.style.Concierge_BenefitsOptIn_ViewPager_PageIndicatorActive
                )
            )
        }
    }

    /**
     * This method should include logic for not allowing the `FlybitsConcierge` show method to be invoked.
     *
     * @param optedIn Whether the user is opted in or out.
     */
    override fun onOptedStateChange(optedIn: Boolean) {
    }

    /**
     * This method will display the Fragment based on the user  opt in and opt out status.
     *
     * @param displayType Depending on the state it will display proper fragment.
     */
    override fun onSwitchFragment(displayType: DisplayType?) {
        when (displayType) {
            DisplayType.SHOW_ERROR -> {
                Logger.appendTag(TAG_BENEFITS).d("User has opted in fail.")
                context?.let { contextNonNull ->
                    if (FlybitsConcierge.with(contextNonNull).optInError.isNullOrBlank())
                        showDialog()
                    else
                        showDialog(
                            msg = FlybitsConcierge.with(contextNonNull).optInError
                        )
                }
                uiOnButtonReset()
            }
            DisplayType.SHOW_CONTENT -> {
                Logger.appendTag(TAG_BENEFITS).d("User has opted in successfully.")
                viewModel.onBenefitsContinueClicked()
            }
            else -> Logger.appendTag(TAG_BENEFITS).d("User has opted in displayType $displayType")
        }
    }

    private fun showDialog(
        title: String = this.getString(R.string.flybits_guided_benefits_error_dialog_title),
        msg: String = this.getString(R.string.flybits_guided_benefits_error_dialog_message)
    ) {
        context?.let { contextNonNull ->
            MaterialDialog.Builder(contextNonNull)
                .title(title)
                .content(msg)
                .positiveText(android.R.string.ok)
                .show()
        }
    }

    override fun onStart() {
        super.onStart()
        FlybitsConcierge.with(context).registerOptedStateChangeListener(this)
    }

    override fun onStop() {
        super.onStop()
        FlybitsConcierge.with(context).unregisterOptedStateChangeListener(this)
    }

    private fun uiOnButtonClick() {
        benefit_progressBar?.visibility = View.VISIBLE

        benefit_fadeBackground.apply {
            this.visibility = View.VISIBLE
            this.animate().alpha(UI_TRANSPARENT)
        }

        flybits_guided_benefits_next.apply {
            this.alpha = UI_TRANSPARENT
            this.isClickable = false

        }
    }

    private fun uiOnButtonReset() {
        benefit_progressBar?.visibility = View.GONE
        benefit_fadeBackground.apply {
            this.visibility = View.GONE
            this.animate().alpha(1f)
        }

        flybits_guided_benefits_next.apply {
            this.alpha = 1f
            this.isClickable = true
        }
    }
}
