package com.flybits.concierge.viewproviders

import android.content.Context
import android.graphics.Color
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewStub
import android.webkit.WebView
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.android.kernel.ExperienceContentAnalytics
import com.flybits.android.kernel.PushContentAnalytics
import com.flybits.commons.library.utils.Utilities
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.R
import com.flybits.concierge.UserActionHandler
import com.flybits.concierge.activities.SimpleCardDetailActivity
import com.flybits.concierge.models.SimpleCard
import com.flybits.concierge.utils.getGlideImageTarget
import com.flybits.concierge.utils.setStreamLineWebViewStyle
import com.flybits.concierge.viewactionhandlers.SimpleCardUserActionHandler
import com.flybits.concierge.viewholders.FlybitsViewHolder

/**
 * [FlybitsViewProvider] for text link cards.
 */
class SimpleCardViewProvider @JvmOverloads internal constructor(
    private val contentAnalytics: ContentAnalytics?,
    private val experienceContentAnalytics: ExperienceContentAnalytics?,
    private val pushContentAnalytics: PushContentAnalytics?,
    private val userActionHandler: UserActionHandler<SimpleCard>,
    private val contentLayoutId: Int = -1
) : FlybitsViewProvider<SimpleCard> {

    // Constructors that are mostly defined for Java compatibility.
    /**
     * Instantiates the [SimpleCardViewProvider] with the default [UserActionHandler] and [contentLayoutId].
     *
     * @param context The [Context] (app context is recommended).
     */
    constructor(context: Context) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        SimpleCardUserActionHandler()
    )

    /**
     * Instantiates the [SimpleCardViewProvider] with provided [UserActionHandler].
     * When overriding only a subset of the default handling logic, extend and override [SimpleCardViewProvider].
     *
     * @param context The [Context] (app context is recommended).
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     */
    constructor(context: Context, userActionHandler: UserActionHandler<SimpleCard>) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        userActionHandler
    )

    /**
     * Instantiates the [SimpleCardViewProvider] with provided [contentLayoutId] and default [UserActionHandler].
     *
     * @param context The [Context] (app context is recommended).
     *
     * @param contentLayoutId The [Int] value that represents content layout ID to be inflated by this [FlybitsViewProvider].
     */
    constructor(context: Context, contentLayoutId: Int) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        SimpleCardUserActionHandler(),
        contentLayoutId
    )

    /**
     * Instantiates the [SimpleCardViewProvider] with provided [UserActionHandler] and [contentLayoutId].
     * When overriding only a subset of the default handling logic, extend and override [SimpleCardViewProvider].
     *
     * @param context The [Context] (app context is recommended).
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     *
     * @param contentLayoutId The [Int] value that represents content layout ID to be inflated by this [FlybitsViewProvider].
     */
    constructor(
        context: Context,
        userActionHandler: UserActionHandler<SimpleCard>,
        contentLayoutId: Int
    ) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        userActionHandler,
        contentLayoutId
    )

    override fun getClassType(): Class<SimpleCard> {
        return SimpleCard::class.java
    }

    override fun getContentType(): String {
        return CONTENT_TYPE
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<SimpleCard> {
        val view = LayoutInflater.from(parent.context)
            .inflate(R.layout.flybits_streamline_item_link, parent, false)
        return SimpleCardViewHolder(
            view,
            contentAnalytics,
            experienceContentAnalytics,
            pushContentAnalytics,
            userActionHandler,
            contentLayoutId
        )
    }

    companion object {
        const val CONTENT_TYPE = "concierge-card-link"
        const val componentID_link = "link"
        const val componentType = "link"

        const val SIMPLE_CARD_TAG = "SimpleCardTag"

        private fun configureView(
            data: SimpleCard,
            simpleCardView: View,
            userActionHandler: UserActionHandler<SimpleCard>,
            flybitsNavigator: FlybitsNavigator,
            contentAnalytics: ContentAnalytics?,
            experienceContentAnalytics: ExperienceContentAnalytics?,
            pushContentAnalytics: PushContentAnalytics?,
            contentLayoutId: Int
        ) {

            val contentView: ViewStub? =
                simpleCardView.findViewById(R.id.flybits_streamline_link_content)

            val emptyView: View =
                simpleCardView.findViewById(R.id.flybits_streamline_link_emptyTxt)

            contentView?.visibility = View.GONE
            emptyView.visibility = View.GONE

            when (data.title?.value) {
                null -> {
                    emptyView.visibility = View.VISIBLE
                }
                else -> {
                    if (contentLayoutId != -1) {
                        contentView?.layoutResource = contentLayoutId
                    }
                    contentView?.visibility = View.VISIBLE

                    val bannerView: ImageView? =
                        simpleCardView.findViewById(R.id.flybits_streamline_link_imgHeader)
                    val headerView: TextView? =
                        simpleCardView.findViewById(R.id.flybits_streamline_link_header)
                    val titleView: TextView? =
                        simpleCardView.findViewById(R.id.flybits_streamline_link_title)
                    val descriptionView: WebView? =
                        simpleCardView.findViewById(R.id.flybits_streamline_link_description)
                    val linkViewProvider: TextView? =
                        simpleCardView.findViewById(R.id.flybits_streamline_link_action)

                    bannerView?.apply {
                        data.image?.value.let {
                            this.visibility = View.GONE
                            if (!it.isNullOrEmpty()) {
                                Glide.with(this.context).load(it)
                                    .apply(RequestOptions().centerCrop())
                                    .into(getGlideImageTarget(this, SIMPLE_CARD_TAG))
                            }
                        }
                    }

                    headerView?.apply {
                        data.header?.value.let {
                            if (it.isNullOrEmpty()) {
                                this.visibility = View.GONE
                            } else {
                                this.text = it
                            }
                        }
                    }

                    headerView?.apply {
                        data.header?.value.let {
                            if (it.isNullOrEmpty()) {
                                this.visibility = View.GONE
                            } else {
                                this.text = it
                            }
                        }
                    }

                    titleView?.text = data.title?.value
                    descriptionView?.apply {
                        data.description?.value.let {
                            when {
                                it.isNullOrEmpty() -> {
                                    this.visibility = View.GONE
                                }
                                else -> {
                                    val rtf = Utilities.toRTF(it)
                                    this.loadDataWithBaseURL(
                                        null,
                                        setStreamLineWebViewStyle(
                                            rtf,
                                            this.context,
                                            R.style.Concierge_StreamLined_Link_Description
                                        ),
                                        "text/html",
                                        "UTF-8",
                                        null
                                    )
                                    this.setBackgroundColor(Color.TRANSPARENT)
                                    this.isVerticalScrollBarEnabled = false
                                }
                            }
                        }
                    }

                    linkViewProvider?.apply {
                        this.text = data.link?.text?.value.toString()
                        this.setOnClickListener {
                            userActionHandler.onUserAction(
                                SimpleCardUserActionHandler.LINK_CLICKED,
                                data,
                                flybitsNavigator
                            )

                            // analytics expected format
                            // ctx.flybits.contentAnalytics.query.componentEngaged.{Content ID}.{Component ID}.{Component Name}.{Component Type}
                            val componentName = this.text.toString().replace(".", "")
                            val key =
                                data.content.id + "." + componentID_link + "." + componentName + "." + componentType
                            contentAnalytics?.trackComponentEngaged(key)
                            contentAnalytics?.trackEngaged(data.content)

                            // ctx.flybits.experienceContentAnalytics.query.componentEngaged.{Content ID}.{Experience ID}.{Component ID}.{Component Name}.{Component Type}
                            data.content.experienceId?.let {
                                val keyExperienceContentAnalytics =
                                    data.content.id + "." + it + "." + componentID_link + "." + componentName + "." + componentType
                                experienceContentAnalytics?.trackComponentEngaged(
                                    keyExperienceContentAnalytics
                                )
                            } ?: run {
                                // ctx.flybits.pushContentAnalytics.query.componentEngaged.{Content ID}.{Push ID}.{Component ID}.{Component Name}.{Component Type}
                                data.content.pushRequestId?.let {
                                    val keyPushContentAnalytics =
                                        data.content.id + "." + it + "." + componentID_link + "." + componentName + "." + componentType
                                    pushContentAnalytics?.trackComponentEngaged(
                                        keyPushContentAnalytics
                                    )
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    class SimpleCardViewHolder(
        val view: View,
        val contentAnalytics: ContentAnalytics?,
        val experienceContentAnalytics: ExperienceContentAnalytics?,
        val pushContentAnalytics: PushContentAnalytics?,
        private val userActionHandler: UserActionHandler<SimpleCard>,
        private val contentLayoutId: Int = -1
    ) : FlybitsViewHolder<SimpleCard>(view) {
        override fun bindData(data: SimpleCard, flybitsNavigator: FlybitsNavigator) {
            configureView(
                data,
                view,
                userActionHandler,
                flybitsNavigator,
                contentAnalytics,
                experienceContentAnalytics,
                pushContentAnalytics,
                contentLayoutId
            )
        }
    }

    override fun onPushOpened(model: SimpleCard, flybitsNavigator: FlybitsNavigator) {
        model.let {
            val bundle = Bundle()
            bundle.putParcelable(SimpleCardDetailActivity.EXTRA_SIMPLECARD_DETAILS, it)
            bundle.putParcelable(
                SimpleCardDetailActivity.EXTRA_USER_ACTION_HANDLER,
                userActionHandler
            )
            flybitsNavigator.openActivity(SimpleCardDetailActivity::class.java, bundle)
        }
    }
}
