package com.flybits.concierge.guidedoptin.data.source

import com.flybits.concierge.guidedoptin.*
import com.flybits.concierge.guidedoptin.data.model.json.*
import com.flybits.concierge.guidedoptin.domain.entity.GuidedOptInEntity
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.io.ByteArrayOutputStream
import java.io.IOException
import java.io.InputStream

const val TAG = "GuidedOptInLocalDataSource"

/**
 * The [GuidedOptInDataSource] implementation for local data source.
 */
internal object GuidedOptInLocalDataSource : GuidedOptInDataSource {
    override suspend fun getGuidedOptInEntity(
        stream: InputStream,
        debug: Boolean
    ): GuidedOptInEntity? {
        return withContext(Dispatchers.IO) {
            try {
                var jsonString = ""
                val outputStream = ByteArrayOutputStream()
                val buf = ByteArray(1024)
                var len: Int
                try {
                    while (stream.read(buf).also { len = it } != -1) {
                        outputStream.write(buf, 0, len)
                    }
                    outputStream.close()
                    stream.close()
                    jsonString = outputStream.toString()
                } catch (e: IOException) {
                    log(debug, TAG, "Failed to read JSON from res/raw/", e)
                }
                log(debug, TAG, "JSON String: $jsonString")

                val guidedOptIn = parseForGuidedOptInModelJSON(jsonString, debug)
                log(debug, TAG, "guidedOptIn: ${guidedOptIn.toString()}")

                guidedOptIn
            } catch (e: Exception) {
                log(debug, TAG, "Failed to fetch and parse JSON from res/raw/", e)
                GuidedOptInModel(
                    BenefitsOptInModel(),
                    LocationOptInModel(),
                    NotificationsOptInModel(),
                    ConciergeOptOutModel()
                )
            }
        }
    }

    fun parseForGuidedOptInModelJSON(jsonString: String, debug: Boolean): GuidedOptInModel? {
        var benefits: BenefitsOptInModel
        var location: LocationOptInModel
        var notifications: NotificationsOptInModel
        var conciergeOptOutModel: ConciergeOptOutModel

        try {
            val config = JSONObject(jsonString)
            // BENEFITS
            try {
                // If key conciergeOptIn is not found - use BenefitsOptInModel with all defaults.
                val benefitsJson = config.getJSONObject("conciergeOptIn")

                val title = try {
                    if (benefitsJson.get("optInTitle") is String) {
                        benefitsJson.getString("optInTitle")
                    } else {
                        benefitsTitleDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get optInTitle", je)
                    benefitsTitleDefault
                }

                val contentJson: JSONArray
                var content: List<BenefitsOptInItemModel>
                try {
                    contentJson = benefitsJson.getJSONArray("pageContent")
                    content = mutableListOf()
                    for (i in 0 until contentJson.length()) {
                        val o = contentJson.getJSONObject(i)

                        val image = try {
                            if (o.get("image") is String) {
                                o.getString("image")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get image", je)
                            ""
                        }

                        val body = try {
                            if (o.get("body") is String) {
                                o.getString("body")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get body", je)
                            ""
                        }
                        content.add(BenefitsOptInItemModel(image, body))
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get pageContent JSONArray", je)
                    content = listOf(
                        BenefitsOptInItemModel(
                            benefitsImageOneDefault,
                            benefitsBodyOneDefault
                        ),
                        BenefitsOptInItemModel(
                            benefitsImageTwoDefault,
                            benefitsBodyTwoDefault
                        ),
                        BenefitsOptInItemModel(
                            benefitsImageThreeDefault,
                            benefitsBodyThreeDefault
                        )
                    )
                }

                val disclaimer = try {
                    if (benefitsJson.get("disclaimerText") is String) {
                        benefitsJson.getString("disclaimerText")
                    } else {
                        benefitsDisclaimerDefault
                    }
                } catch (je: Exception) {
                    log(debug, TAG, "Failed to get disclaimerText", je)
                    benefitsDisclaimerDefault
                }

                val disclaimerCheckBox = try {
                    if (benefitsJson.get("mandatoryDisclaimerCheckBox") is Boolean) {
                        benefitsJson.getBoolean("mandatoryDisclaimerCheckBox")
                    } else {
                        true
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get mandatoryDisclaimerCheckBox", je)
                    true
                }

                val buttonTitle = try {
                    if (benefitsJson.get("optInButtonTitle") is String) {
                        benefitsJson.getString("optInButtonTitle")
                    } else {
                        guidedContinueDefault
                    }

                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get optInButtonTitle", je)
                    guidedContinueDefault
                }

                benefits =
                    BenefitsOptInModel(title, content, disclaimer, disclaimerCheckBox, buttonTitle)

            } catch (je: JSONException) {
                log(debug, TAG, "Failed to get conciergeOptIn JSONObject", je)
                benefits = BenefitsOptInModel()
            }

            // LOCATION SERVICES
            try {
                // If key locationServices is not found - use LocationOptInModel with all defaults.
                val locationJson = config.getJSONObject("locationServices")

                val exclude = try {
                    if (locationJson.get("exclude") is Boolean) {
                        locationJson.getBoolean("exclude")
                    } else {
                        false
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get exclude", je)
                    false
                }

                val title = try {
                    if (locationJson.get("locationServiceTitle") is String) {
                        locationJson.getString("locationServiceTitle")
                    } else {
                        locationTitleDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get locationServiceTitle", je)
                    locationTitleDefault
                }

                val locationInfoJson: JSONArray
                var info: List<LocationOptInItemModel>
                try {
                    locationInfoJson = locationJson.getJSONArray("infoList")
                    info = mutableListOf()
                    for (i in 0 until locationInfoJson.length()) {
                        val o = locationInfoJson.getJSONObject(i)

                        val image = try {
                            if (o.get("image") is String) {
                                o.getString("image")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get image", je)
                            ""
                        }

                        val body = try {
                            if (o.get("body") is String) {
                                o.getString("body")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get body", je)
                            ""
                        }
                        info.add(LocationOptInItemModel(image, body))
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get infoList JSONArray", je)
                    info = listOf(
                        LocationOptInItemModel(
                            locationImageOneDefault,
                            locationBodyOneDefault
                        ),
                        LocationOptInItemModel(
                            locationImageTwoDefault,
                            locationBodyTwoDefault
                        ),
                        LocationOptInItemModel(
                            locationImageThreeDefault,
                            locationBodyThreeDefault
                        )
                    )
                }

                val next = try {
                    if (locationJson.get("nextButtonTitle") is String) {
                        locationJson.getString("nextButtonTitle")
                    } else {
                        guidedContinueDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get nextButtonTitle", je)
                    guidedContinueDefault
                }

                val skip = try {
                    if (locationJson.get("skipButtonTitle") is String) {
                        locationJson.getString("skipButtonTitle")
                    } else {
                        guidedSkipDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get skipButtonTitle", je)
                    guidedSkipDefault
                }

                location = LocationOptInModel(exclude, title, info, next, skip)
            } catch (je: JSONException) {
                log(debug, TAG, "Failed to get locationServices JSONObject", je)
                location = LocationOptInModel()
            }

            // NOTIFICATIONS
            try {
                // If key notifications is not found - use NotificationsOptInModel with all defaults.
                val notificationsJson = config.getJSONObject("notifications")

                val exclude = try {
                    if (notificationsJson.get("exclude") is Boolean) {
                        notificationsJson.getBoolean("exclude")
                    } else {
                        false
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get exclude", je)
                    false
                }

                val title = try {
                    if (notificationsJson.get("notificationsTitle") is String) {
                        notificationsJson.getString("notificationsTitle")
                    } else {
                        notificationsTitleDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get notificationsTitle", je)
                    notificationsTitleDefault
                }

                val notificationsInfoJson: JSONArray
                var info: List<NotificationsOptInItemModel>
                try {
                    notificationsInfoJson = notificationsJson.getJSONArray("infoList")
                    info = mutableListOf()
                    for (i in 0 until notificationsInfoJson.length()) {
                        val o = notificationsInfoJson.getJSONObject(i)

                        val image = try {
                            if (o.get("image") is String) {
                                o.getString("image")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get image", je)
                            ""
                        }

                        val body = try {
                            if (o.get("body") is String) {
                                o.getString("body")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get body", je)
                            ""
                        }
                        info.add(NotificationsOptInItemModel(image, body))
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get infoList JSONArray", je)
                    info = listOf(
                        NotificationsOptInItemModel(
                            notificationsImageOneDefault,
                            notificationsBodyOneDefault
                        ),
                        NotificationsOptInItemModel(
                            notificationsImageTwoDefault,
                            notificationsBodyTwoDefault
                        ),
                        NotificationsOptInItemModel(
                            notificationsImageThreeDefault,
                            notificationsBodyThreeDefault
                        )
                    )
                }

                val disclaimerText = try {
                    if (notificationsJson.get("disclaimerText") is String) {
                        notificationsJson.getString("disclaimerText")
                    } else {
                        notificationsDisclaimerTextDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG,"Failed to get disclaimerText", je)
                    notificationsDisclaimerTextDefault
                }

                val next = try {
                    if (notificationsJson.get("nextButtonTitle") is String) {
                        notificationsJson.getString("nextButtonTitle")
                    } else {
                        guidedContinueDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get nextButtonTitle", je)
                    guidedContinueDefault
                }

                val skip = try {
                    if (notificationsJson.get("skipButtonTitle") is String) {
                        notificationsJson.getString("skipButtonTitle")
                    } else {
                        guidedSkipDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get skipButtonTitle", je)
                    guidedSkipDefault
                }

                notifications = NotificationsOptInModel(exclude, title, info, disclaimerText, next, skip)
            } catch (je: JSONException) {
                log(debug, TAG, "Failed to get notifications JSONObject", je)
                notifications = NotificationsOptInModel()
            }

            // Concierge Opt Out
            try {
                // If key notifications is not found - use NotificationsOptInModel with all defaults.
                val conciergeOptOutJson = config.getJSONObject("conciergeOptOut")

                val title = try {
                    if (conciergeOptOutJson.get("optOutTitle") is String) {
                        conciergeOptOutJson.getString("optOutTitle")
                    } else {
                        notificationsTitleDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get notificationsTitle", je)
                    notificationsTitleDefault
                }

                val conciergeOptOutInfoJson: JSONArray
                var info: List<ConciergeOptOutItemModel>
                try {
                    conciergeOptOutInfoJson = conciergeOptOutJson.getJSONArray("infoList")
                    info = mutableListOf()
                    for (i in 0 until conciergeOptOutInfoJson.length()) {
                        val o = conciergeOptOutInfoJson.getJSONObject(i)

                        val image = try {
                            if (o.get("image") is String) {
                                o.getString("image")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get image", je)
                            ""
                        }

                        val body = try {
                            if (o.get("body") is String) {
                                o.getString("body")
                            } else {
                                ""
                            }
                        } catch (je: JSONException) {
                            log(debug, TAG, "Failed to get body", je)
                            ""
                        }
                        info.add(ConciergeOptOutItemModel(image, body))
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get infoList JSONArray", je)
                    info = listOf(
                        ConciergeOptOutItemModel(
                            notificationsImageOneDefault,
                            notificationsBodyOneDefault
                        ),
                        ConciergeOptOutItemModel(
                            notificationsImageTwoDefault,
                            notificationsBodyTwoDefault
                        ),
                        ConciergeOptOutItemModel(
                            notificationsImageThreeDefault,
                            notificationsBodyThreeDefault
                        )
                    )
                }

                val next = try {
                    if (conciergeOptOutJson.get("optOutButtonTitle") is String) {
                        conciergeOptOutJson.getString("optOutButtonTitle")
                    } else {
                        guidedContinueDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get optOutButtonTitle", je)
                    guidedContinueDefault
                }

                val skip = try {
                    if (conciergeOptOutJson.get("cancelButtonTitle") is String) {
                        conciergeOptOutJson.getString("cancelButtonTitle")
                    } else {
                        guidedSkipDefault
                    }
                } catch (je: JSONException) {
                    log(debug, TAG, "Failed to get cancelButtonTitle", je)
                    guidedSkipDefault
                }

                conciergeOptOutModel = ConciergeOptOutModel(title, info, next, skip)
            } catch (je: JSONException) {
                conciergeOptOutModel = ConciergeOptOutModel()
            }

            return GuidedOptInModel(
                benefits,
                location,
                notifications,
                conciergeOptOutModel
            )
        } catch (je: JSONException) {
            log(debug, TAG, "Failed to get Guided Opt-In JSON configuration.", je)
            return GuidedOptInModel(
                BenefitsOptInModel(),
                LocationOptInModel(),
                NotificationsOptInModel(),
                ConciergeOptOutModel()
            )
        }
    }
}