package com.flybits.concierge.fragments

import android.net.http.SslError
import android.os.Build
import android.os.Bundle
import android.support.v4.app.Fragment
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.*
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.R
import kotlinx.android.synthetic.main.flybits_con_fragment_document.*
import kotlinx.android.synthetic.main.flybits_con_fragment_offline.*

/**
 * This fragment is responsible for displaying the terms and conditions or Privacy policy documents.
 * You can provide url and/or text.
 * if text is provided a fragment with the text will be displayed.
 * if url is provided a fragment with a web-page will be displayed.
 * Instantiate this fragment using the exposed `newInstance(url,text)` method.
 */

class DocumentFragment : Fragment() {

    private var loading: Boolean = false
    private var errorPresent: Boolean = false
    private var url: String = ""
    private var text: String? = null


    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        return inflater.inflate(R.layout.flybits_con_fragment_document, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        url = arguments?.getString(EXTRA_URL) ?: ""
        text = arguments?.getString(EXTRA_TEXT)

        if (savedInstanceState == null) {
            if (!text.isNullOrEmpty()) {
                flybits_con_document_fragment_prgLoading?.visibility = View.GONE
                flybits_con_fragment_document_webBody?.visibility = View.GONE
                flybits_con_fragment_document_scrollview?.visibility = View.VISIBLE
                flybits_con_fragment_document_text?.text = text
            } else if (URLUtil.isValidUrl(url)) {
                flybits_con_fragment_document_scrollview?.visibility = View.GONE
                flybits_con_fragment_document_text?.visibility = View.GONE
                loadWebView(url)
            } else {
                Logger.e("DocumentFragment: Invalid URL provided")
            }
        } else restoreWebView(savedInstanceState)
    }

    private fun restoreWebView(savedInstanceState: Bundle) {
        loading = savedInstanceState.getBoolean(ConciergeConstants.STATE_LOADING)
        url = savedInstanceState.getString(ConciergeConstants.STATE_URL) ?: ""

        if (loading) {
            loading = false
            loadWebView(url)
        } else {
            flybits_con_fragment_document_webBody?.restoreState(savedInstanceState)
            errorPresent = savedInstanceState.getBoolean(ConciergeConstants.STATE_ERROR_PRESENT)
            if (errorPresent) {
                flybits_con_fragment_document_error_holder?.visibility = View.VISIBLE
                flybits_con_fragment_document_webBody?.visibility = View.GONE
                offline_fragment_errText?.text =
                    resources.getString(R.string.flybits_streamline_invalid_url)
                offline_fragment_retry_button?.visibility = View.GONE
            } else {
                flybits_con_fragment_document_error_holder?.visibility = View.GONE
                flybits_con_fragment_document_webBody?.visibility = View.VISIBLE
            }
        }
    }

    private fun loadWebView(url: String) {
        if (loading) return
        loading = true
        errorPresent = false
        flybits_con_fragment_document_webBody?.loadUrl(url)
        flybits_con_document_fragment_prgLoading?.visibility = View.VISIBLE
        flybits_con_fragment_document_webBody?.visibility = View.GONE
        flybits_con_fragment_document_webBody.settings.apply {
            this.cacheMode = WebSettings.LOAD_NO_CACHE
            this.javaScriptEnabled = true
            this.domStorageEnabled = true
        }
        flybits_con_fragment_document_webBody?.webViewClient = object : WebViewClient() {
            override fun onPageFinished(view: WebView, url: String) {
                loading = false
                flybits_con_document_fragment_prgLoading?.visibility = View.GONE
                if (errorPresent) {
                    flybits_con_fragment_document_webBody?.visibility = View.GONE
                    flybits_con_fragment_document_error_holder?.visibility = View.VISIBLE
                    offline_fragment_errText?.text =
                        resources.getString(R.string.flybits_streamline_invalid_url)
                    offline_fragment_retry_button?.visibility = View.GONE
                } else {
                    flybits_con_fragment_document_error_holder?.visibility = View.VISIBLE
                    flybits_con_fragment_document_webBody?.visibility = View.VISIBLE
                }
                super.onPageFinished(view, url)
            }

            override fun onReceivedError(
                view: WebView,
                request: WebResourceRequest?,
                error: WebResourceError
            ) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    Logger.d(DocumentFragment::class.java.simpleName + ": error: " + error.description.toString() + ", code: " + error.errorCode)
                } else {
                    Logger.d(DocumentFragment::class.java.simpleName + ": error: " + error.toString())
                }
                errorPresent = true
                super.onReceivedError(view, request, error)
            }

            override fun onReceivedHttpError(
                view: WebView,
                request: WebResourceRequest?,
                errorResponse: WebResourceResponse?
            ) {
                if (view.measuredHeightAndState <= 0)
                    errorPresent = true
                super.onReceivedHttpError(view, request, errorResponse)
            }

            override fun onReceivedSslError(
                view: WebView,
                handler: SslErrorHandler?,
                error: SslError?
            ) {
                if (view.measuredHeightAndState <= 0)
                    errorPresent = true
                super.onReceivedSslError(view, handler, error)
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        flybits_con_fragment_document_webBody?.saveState(outState)
        outState.putBoolean(ConciergeConstants.STATE_LOADING, loading)
        outState.putBoolean(ConciergeConstants.STATE_ERROR_PRESENT, errorPresent)
        outState.putString(ConciergeConstants.STATE_URL, url)
    }

    companion object {
        const val EXTRA_URL = "flybits_con_extra_document_url"
        const val EXTRA_TEXT = "flybits_con_extra_document_text"

        fun newInstance(url: String, text: String?): DocumentFragment {
            val bundle = Bundle()
            val fragment = DocumentFragment()
            bundle.putString(EXTRA_URL, url)
            bundle.putString(EXTRA_TEXT, text)
            fragment.arguments = bundle
            return fragment
        }
    }
}