package com.flybits.concierge.guidedoptin.presentation.view

import android.Manifest
import android.arch.lifecycle.ViewModelProviders
import android.os.Bundle
import android.support.v4.app.DialogFragment
import android.support.v4.app.Fragment
import android.support.v4.app.FragmentActivity
import android.support.v7.widget.LinearLayoutManager
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.TextView
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.R
import com.flybits.concierge.guidedoptin.GuidedOptInFlowListener
import com.flybits.concierge.guidedoptin.presentation.adapter.LocationOptInAdapter
import com.flybits.concierge.guidedoptin.presentation.viewmodel.GuidedOptInViewModel
import com.flybits.concierge.utils.isPermissionGranted
import kotlinx.android.synthetic.main.flybits_fragment_guided_location.*

/**
 * This LocationOptInFragment is responsible for Location Guided Opt In UI.
 */
internal class LocationOptInFragment : Fragment(), GuidedOptInFlowListener {

    private val viewModel: GuidedOptInViewModel by lazy {
        ViewModelProviders.of(activity!!).get(GuidedOptInViewModel::class.java)
    }

    private var isContinueClicked = false

    // Set to true if openDialogFragment is called as callback.
    private var isDialogFragmentVisible = false

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? = inflater.inflate(R.layout.flybits_fragment_guided_location, container, false)

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val supportActionBarTitle: TextView? =
            activity?.findViewById(R.id.flybits_guided_toolbar_title)
        supportActionBarTitle?.text = getString(R.string.flybits_guided_location_appbar_title)

        viewModel.getLocationOptIn()?.let {
            flybits_guided_location_title.text = getString(it.title)
            flybits_guided_location_next.text = getString(it.nextTitle)
            flybits_guided_location_skip.text = getString(it.skipTitle)

            if (it.info.isNotEmpty()) {
                flybits_guided_location_rv.apply {
                    layoutManager = LinearLayoutManager(context)
                    adapter = LocationOptInAdapter(it.info)
                    addItemDecoration(
                        MarginItemDecoration(
                            resources.getDimension(R.dimen.flybits_content_margin_xxlarge)
                                .toInt(), it.info.size
                        )
                    )
                }
            }
        }

        // Set listener for Continue/next button
        flybits_guided_location_next.setOnClickListener {
            isContinueClicked = true
            context?.let {
                viewModel.onLocationContinueClicked(
                    isPermissionGranted(
                        it,
                        Manifest.permission.ACCESS_FINE_LOCATION
                    )
                )
            }
        }

        // Set listener for skip button
        flybits_guided_location_skip.setOnClickListener {
            viewModel.onLocationSkipClicked()
        }
    }

    override fun onResume() {
        super.onResume()

        // Add this Fragment to GuidedOptInFlowListener.
        FlybitsConcierge.with(context).guidedOptInFlowListener?.let {
            synchronized(it) {
                it.add(
                    this
                )
            }
        }

        // Set skip button visibility based on first level of permission for location.
        context?.let {
            if (isPermissionGranted(it, Manifest.permission.ACCESS_FINE_LOCATION)) {
                flybits_guided_location_skip.visibility = View.GONE
            }
        }

        /*Allows use to move onto next screen once location permission
          dialog (triggered by initial continue button click) is dismissed (dismissal of dialog triggers fragment onResume)
         */
        activity?.let {
            if (!hasOpenedDialogs(it)) {
                if (isContinueClicked && !isDialogFragmentVisible) {
                    isContinueClicked = false
                    viewModel.onLocationContinueClicked(true)
                }
            }
        }
    }

    /*
    *  Check if DialogFragment is displayed on top.
    */
    fun hasOpenedDialogs(activity: FragmentActivity): Boolean {
        val fragments = activity.supportFragmentManager.fragments
        for (fragment in fragments) {
            if (fragment is DialogFragment) {
                isContinueClicked = false
                return true
            }
        }
        return false
    }

    override fun onPause() {
        super.onPause()
        //Remove this Fragment to GuidedOptInFlowListener.
        FlybitsConcierge.with(context).guidedOptInFlowListener?.let {
            synchronized(it) {
                it.remove(
                    this
                )
            }
        }
    }

    companion object {
        fun newInstance() = LocationOptInFragment()
    }

    override fun openDialogFragment(dialog: DialogFragment) {
        isDialogFragmentVisible = true
        dialog.show(activity?.supportFragmentManager, "")
    }
}