package com.flybits.concierge.viewproviders

import android.content.Context
import android.graphics.Color
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewStub
import android.webkit.WebView
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.android.kernel.ExperienceContentAnalytics
import com.flybits.android.kernel.PushContentAnalytics
import com.flybits.commons.library.utils.Utilities
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.R
import com.flybits.concierge.UserActionHandler
import com.flybits.concierge.activities.ActionCardDetailActivity
import com.flybits.concierge.models.ActionCard
import com.flybits.concierge.utils.getGlideImageTarget
import com.flybits.concierge.utils.setStreamLineWebViewStyle
import com.flybits.concierge.viewactionhandlers.ActionCardUserActionHandler
import com.flybits.concierge.viewholders.FlybitsViewHolder


/**
 * [FlybitsViewProvider] for button link cards.
 */
class ActionCardViewProvider internal constructor(
    private val contentAnalytics: ContentAnalytics?,
    private val experienceContentAnalytics: ExperienceContentAnalytics?,
    private val pushContentAnalytics: PushContentAnalytics?,
    private val userActionHandler: UserActionHandler<ActionCard>,
    private val contentLayoutId: Int = -1
) : FlybitsViewProvider<ActionCard> {

    // Constructors that are mostly defined for Java compatibility.
    /**
     * Instantiates the [ActionCardViewProvider] with the default [UserActionHandler] and [contentLayoutId].
     *
     * @param context The [Context] (app context is recommended).
     */
    constructor(context: Context) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        ActionCardUserActionHandler()
    )

    /**
     * Instantiates the [ActionCardViewProvider] with provided [UserActionHandler].
     * When overriding only a subset of the default handling logic, extend and override [ActionCardViewProvider].
     *
     * @param context The [Context] (app context is recommended).
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     */
    constructor(context: Context, userActionHandler: UserActionHandler<ActionCard>) : this(
        ContentAnalytics(context), ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        userActionHandler
    )

    /**
     * Instantiates the [ActionCardViewProvider] with provided [contentLayoutId] and default [UserActionHandler].
     *
     * @param context The [Context] (app context is recommended).
     *
     * @param contentLayoutId The [Int] value that represents content layout ID to be inflated by this [FlybitsViewProvider].
     */
    constructor(context: Context, contentLayoutId: Int) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context),
        PushContentAnalytics(context),
        ActionCardUserActionHandler(),
        contentLayoutId
    )

    /**
     * Instantiates the [ActionCardViewProvider] with provided [UserActionHandler] and [contentLayoutId].
     * When overriding only a subset of the default handling logic, extend and override [ActionCardViewProvider].
     *
     * @param context The [Context] (app context is recommended).
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     *
     * @param contentLayoutId The [Int] value that represents content layout ID to be inflated by this [FlybitsViewProvider].
     */
    constructor(
        context: Context,
        userActionHandler: UserActionHandler<ActionCard>,
        contentLayoutId: Int
    ) : this(
        ContentAnalytics(context),
        ExperienceContentAnalytics(context), PushContentAnalytics(context),
        userActionHandler,
        contentLayoutId
    )

    override fun getClassType(): Class<ActionCard> {
        return ActionCard::class.java
    }

    override fun getContentType(): String {
        return CONTENT_TYPE
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<ActionCard> {
        val view = LayoutInflater.from(parent.context)
            .inflate(R.layout.flybits_streamline_item_button, parent, false)
        return ActionCardViewHolder(
            view,
            contentAnalytics,
            experienceContentAnalytics,
            pushContentAnalytics,
            userActionHandler,
            contentLayoutId
        )
    }

    companion object {
        const val CONTENT_TYPE = "concierge-card-buttons"
        const val componentID_btn1 = "btn1"
        const val componentID_btn2 = "btn2"
        const val componentType = "buttons"

        const val ACTION_CARD_TAG = "ActionCardTag"


        private fun configureView(
            data: ActionCard,
            actionCardView: View,
            userActionHandler: UserActionHandler<ActionCard>,
            flybitsNavigator: FlybitsNavigator,
            contentAnalytics: ContentAnalytics?,
            experienceContentAnalytics: ExperienceContentAnalytics?,
            pushContentAnalytics: PushContentAnalytics?,
            contentLayoutId: Int
        ) {
            val contentView: ViewStub? =
                actionCardView.findViewById(R.id.flybits_streamline_button_content)

            val emptyView: View =
                actionCardView.findViewById(R.id.flybits_streamline_button_emptyTxt)

            contentView?.visibility = View.GONE
            emptyView.visibility = View.GONE

            when (data.title?.value) {
                null -> {
                    emptyView.visibility = View.VISIBLE
                }
                else -> {
                    if (contentLayoutId != -1) {
                        contentView?.layoutResource = contentLayoutId
                    }
                    contentView?.visibility = View.VISIBLE

                    val bannerView: ImageView? =
                        actionCardView.findViewById(R.id.flybits_streamline_button_imgHeader)
                    val headerView: TextView? =
                        actionCardView.findViewById(R.id.flybits_streamline_button_header)
                    val titleView: TextView? =
                        actionCardView.findViewById(R.id.flybits_streamline_button_title)
                    val descriptionWebView: WebView? =
                        actionCardView.findViewById(R.id.flybits_streamline_button_description)
                    val primaryButton: Button? =
                        actionCardView.findViewById(R.id.flybits_streamline_button_primary_action)
                    val secondaryButton: Button? =
                        actionCardView.findViewById(R.id.flybits_streamline_button_secondary_button)

                    bannerView?.apply {
                        data.image?.value.let {
                            this.visibility = View.GONE
                            if (!it.isNullOrEmpty()) {
                                Glide.with(this.context).load(it)
                                    .apply(RequestOptions().centerCrop())
                                    .into(
                                        getGlideImageTarget(
                                            this,
                                            ACTION_CARD_TAG
                                        )
                                    )
                            }
                        }
                    }

                    headerView?.apply {
                        data.header?.value.let {
                            if (it.isNullOrEmpty()) {
                                this.visibility = View.GONE
                            } else {
                                this.text = it
                            }
                        }
                    }

                    titleView?.text = data.title?.value

                    descriptionWebView?.apply {
                        data.description?.value.let {
                            when {
                                it.isNullOrEmpty() -> {
                                    this.visibility = View.GONE
                                }
                                else -> {
                                    val rtf = Utilities.toRTF(it)
                                    this.loadDataWithBaseURL(
                                        null,
                                        setStreamLineWebViewStyle(
                                            rtf,
                                            this.context,
                                            R.style.Concierge_StreamLined_Button_Description
                                        ),
                                        "text/html",
                                        "UTF-8",
                                        null
                                    )
                                    this.setBackgroundColor(Color.TRANSPARENT)
                                    this.isVerticalScrollBarEnabled = false
                                }
                            }
                        }
                    }

                    when (data.buttons?.list?.size) {
                        0 -> {
                            primaryButton?.visibility = View.GONE
                            secondaryButton?.visibility = View.GONE
                        }
                        1 -> {
                            primaryButton.let {
                                it?.visibility = View.VISIBLE
                                it?.text = data.buttons?.list?.get(0)?.text?.value.toString()
                            }
                            secondaryButton?.visibility = View.GONE
                        }
                        2 -> {
                            primaryButton.let {
                                it?.visibility = View.VISIBLE
                                it?.text = data.buttons?.list?.get(0)?.text?.value.toString()
                            }
                            secondaryButton.let {
                                it?.visibility = View.VISIBLE
                                it?.text = data.buttons?.list?.get(1)?.text?.value.toString()
                            }
                        }
                        else -> {
                            primaryButton?.visibility = View.GONE
                            secondaryButton?.visibility = View.GONE
                        }
                    }

                    primaryButton?.setOnClickListener {
                        userActionHandler.onUserAction(
                            ActionCardUserActionHandler.PRIMARY_CLICKED,
                            data,
                            flybitsNavigator
                        )

                        // analytics expected format
                        // ctx.flybits.contentAnalytics.query.componentEngaged.{Content ID}.{Component ID}.{Component Name}.{Component Type}
                        val size = data.buttons?.list?.size
                        if (size != null && size >= 1) {
                            val componentName =
                                data.buttons?.list?.get(0)?.text?.value.toString()
                                    .replace(".", "")
                            val key =
                                data.content.id + "." + componentID_btn1 + "." + componentName + "." + componentType
                            contentAnalytics?.trackComponentEngaged(key)
                            contentAnalytics?.trackEngaged(data.content)

                            // ctx.flybits.experienceContentAnalytics.query.componentEngaged.{Content ID}.{Experience ID}.{Component ID}.{Component Name}.{Component Type}
                            data.content.experienceId?.let {
                                val keyExperienceContentAnalytics =
                                    data.content.id + "." + it + "." + componentID_btn1 + "." + componentName + "." + componentType
                                experienceContentAnalytics?.trackComponentEngaged(
                                    keyExperienceContentAnalytics
                                )
                            } ?: run {
                                // ctx.flybits.pushContentAnalytics.query.componentEngaged.{Content ID}.{Push ID}.{Component ID}.{Component Name}.{Component Type}
                                data.content.pushRequestId?.let {
                                    val keyPushContentAnalytics =
                                        data.content.id + "." + it + "." + componentID_btn1 + "." + componentName + "." + componentType
                                    pushContentAnalytics?.trackComponentEngaged(
                                        keyPushContentAnalytics
                                    )
                                }
                            }
                        }
                    }
                    secondaryButton?.setOnClickListener {
                        userActionHandler.onUserAction(
                            ActionCardUserActionHandler.SECONDARY_CLICKED,
                            data,
                            flybitsNavigator
                        )
                        val size = data.buttons?.list?.size
                        if (size != null && size >= 2) {
                            val componentName =
                                data.buttons?.list?.get(1)?.text?.value.toString()
                                    .replace(".", "")
                            val key =
                                data.content.id + "." + componentID_btn2 + "." + componentName + "." + componentType
                            contentAnalytics?.trackComponentEngaged(key)
                            contentAnalytics?.trackEngaged(data.content)

                            // ctx.flybits.experienceContentAnalytics.query.componentEngaged.{Content ID}.{Experience ID}.{Component ID}.{Component Name}.{Component Type}
                            data.content.experienceId?.let {
                                val keyExperienceContentAnalytics =
                                    data.content.id + "." + it + "." + componentID_btn2 + "." + componentName + "." + componentType
                                experienceContentAnalytics?.trackComponentEngaged(
                                    keyExperienceContentAnalytics
                                )
                            } ?: run {
                                // ctx.flybits.pushContentAnalytics.query.componentEngaged.{Content ID}.{Push ID}.{Component ID}.{Component Name}.{Component Type}
                                data.content.pushRequestId?.let {
                                    val keyPushContentAnalytics =
                                        data.content.id + "." + it + "." + componentID_btn2 + "." + componentName + "." + componentType
                                    pushContentAnalytics?.trackComponentEngaged(
                                        keyPushContentAnalytics
                                    )
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    class ActionCardViewHolder(
        val view: View,
        val contentAnalytics: ContentAnalytics?,
        val experienceContentAnalytics: ExperienceContentAnalytics?,
        val pushContentAnalytics: PushContentAnalytics?,
        private val userActionHandler: UserActionHandler<ActionCard>,
        private val contentLayoutId: Int = -1
    ) : FlybitsViewHolder<ActionCard>(view) {
        override fun bindData(data: ActionCard, flybitsNavigator: FlybitsNavigator) {
            configureView(
                data,
                view,
                userActionHandler,
                flybitsNavigator,
                contentAnalytics,
                experienceContentAnalytics,
                pushContentAnalytics,
                contentLayoutId
            )
        }
    }

    override fun onPushOpened(model: ActionCard, flybitsNavigator: FlybitsNavigator) {
        model.let {
            val bundle = Bundle()
            bundle.putParcelable(ActionCardDetailActivity.EXTRA_ACTIONCARD_DETAILS, it)
            bundle.putParcelable(
                ActionCardDetailActivity.EXTRA_USER_ACTION_HANDLER,
                userActionHandler
            )
            flybitsNavigator.openActivity(ActionCardDetailActivity::class.java, bundle)
        }
    }
}
