package com.flybits.concierge.guidedoptin.presentation.view

import android.arch.lifecycle.Observer
import android.arch.lifecycle.ViewModelProviders
import android.os.Bundle
import android.support.v4.app.Fragment
import android.support.v7.app.AppCompatActivity
import android.view.MenuItem
import android.view.View
import com.afollestad.materialdialogs.MaterialDialog
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.InternalPreferences
import com.flybits.concierge.R
import com.flybits.concierge.guidedoptin.presentation.model.GuidedOptIn
import com.flybits.concierge.guidedoptin.presentation.viewmodel.GuidedOptInViewModel
import com.flybits.concierge.guidedoptin.presentation.viewmodel.GuidedOptInViewModelFactory
import kotlinx.android.synthetic.main.flybits_activity_guided.*

const val RESOURCE_NAME = "guided_opt_in_config"
const val RESOURCE_RAW_TYPE = "raw"

/**
 * This GuidedOptInActivity is responsible for holding the Guided Opt In fragments.
 */
internal class GuidedOptInActivity : AppCompatActivity() {
    private val viewModel: GuidedOptInViewModel by lazy {
        ViewModelProviders.of(
            this,
            GuidedOptInViewModelFactory(FlybitsConcierge.with(this))
        )
            .get(GuidedOptInViewModel::class.java)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.flybits_activity_guided)
        setSupportActionBar(findViewById(R.id.flybits_guided_toolbar))
        supportActionBar?.apply {
            setHomeButtonEnabled(true)
            setDisplayHomeAsUpEnabled(true)
            setDisplayShowTitleEnabled(false)
        }

        flybits_guided_progress_bar.visibility = View.VISIBLE

        val isOptInActivity = intent.extras?.getBoolean(IS_OPT_IN, true) ?: true

        val res = resources.getIdentifier(RESOURCE_NAME, RESOURCE_RAW_TYPE, packageName)
        viewModel.fetchGuidedOptIn(resources, res, packageName)
        viewModel.guidedOptIn.observe(this, Observer<GuidedOptIn> { guidedOptIn ->
            guidedOptIn?.let {
                flybits_guided_progress_bar.visibility = View.GONE
                if (isOptInActivity) {
                    val fragment = BenefitsOptInFragment.newInstance()
                    openFragment(fragment)
                } else { //launch opt-out flow instead
                    FlybitsConcierge.with(this)
                        .isOptedInLocal(object : ObjectResultCallback<Boolean> {
                            /**
                             * Indicates that the request failed.
                             *
                             * @param exception The [FlybitsException] that indicates the exception that was thrown.
                             */
                            override fun onException(exception: FlybitsException) {
                                openSimpleDialog(
                                    getString(R.string.flybits_opt_out_error_dialog_title),
                                    getString(R.string.flybits_opt_out_error_dialog_message)
                                )
                            }

                            /**
                             * Indicates that the request was successfully made.
                             *
                             * @param item The item that is expected to be returned.
                             */
                            override fun onSuccess(item: Boolean) {
                                if (item) {
                                    openFragment(OptOutFragment.newInstance())
                                } else {
                                    openSimpleDialog(
                                        getString(R.string.flybits_opt_out_already_dialog_title),
                                        getString(R.string.flybits_opt_out_already_dialog_message)
                                    )
                                }
                            }

                        })
                }
            }
        })

        viewModel.navigateToLocationOptInFragment.observe(this, Observer {
            if (it!!) {
                openFragment(LocationOptInFragment.newInstance())
                viewModel.navigateToLocationOptInFragmentCompleted()
            }
        })

        viewModel.navigateToNotificationsOptInFragment.observe(this, Observer {
            if (it!!) {
                openFragment(NotificationsOptInFragment.newInstance())
                viewModel.navigateToNotificationsOptInFragmentCompleted()
            }
        })

        viewModel.finishGuidedOptInFlow.observe(this, Observer {
            if (it!!) {
                finish()
                viewModel.finishGuidedOptInFlowCompleted()
            }
        })
    }

    private fun openFragment(fragment: Fragment) {
        supportFragmentManager
            .beginTransaction()
            .replace(
                R.id.flybits_guided_frameLayout, fragment
            )
            .commit()
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return if (item.itemId == android.R.id.home) {
            onBackPressed()
            true
        } else {
            super.onOptionsItemSelected(item)
        }
    }

    private fun openSimpleDialog(title: String, message: String) {
        MaterialDialog.Builder(this)
            .title(title)
            .content(message)
            .positiveText(R.string.flybits_con_ok)
            .onPositive { _, _ ->
                finish()
                viewModel.finishGuidedOptInFlowCompleted()
            }
            .dismissListener {
                finish()
                viewModel.finishGuidedOptInFlowCompleted()
            }
            .show()
    }

    companion object {
        const val IS_OPT_IN = "flybits_extra_is_opt_in"
    }

    override fun onBackPressed() {
        super.onBackPressed()
        InternalPreferences.setLaunchedFromGuidedOptInActivity(applicationContext, true)
    }
}