package com.flybits.concierge.guidedoptin.presentation.mapper

import android.content.res.Resources
import com.flybits.concierge.BuildConfig
import com.flybits.concierge.guidedoptin.*
import com.flybits.concierge.guidedoptin.domain.entity.*
import com.flybits.concierge.guidedoptin.presentation.model.*

const val TAG = "DataToPresentationMapper"
const val RESOURCE_TYPE_DRAWABLE = "drawable"
const val RESOURCE_TYPE_STRING = "string"

/**
 * The mapper class to provide mapping between entities of domain layer and models that serve for providing processed data in presentation layer.
 */
internal class DataToPresentationMapper {
    /**
     * Maps [GuidedOptInEntity] to presentation layer [GuidedOptIn] model who's data provides references to resources that are ready for binding to Views.
     */
    fun map(res: Resources, packageName: String, model: GuidedOptInEntity): GuidedOptIn {
        val guidedOptIn = GuidedOptIn(
            map(res, packageName, model.benefits),
            map(res, packageName, model.location),
            map(res, packageName, model.notifications),
            map(res, packageName, model.conciergeOptOut)
        )
        log(BuildConfig.DEBUG, TAG, "Presentation Guided Opt-In model: $guidedOptIn")

        return guidedOptIn
    }

    /**
     * Maps [BenefitsOptInEntity] to presentation layer [BenefitsOptIn] model who's data provides references to resources that are ready for binding to Views.
     * Validation on properties of [BenefitsOptInEntity] in this function is done to ensure that values for valid keys in guided_opt_in_config.json will always reference to existing string resources.
     * Image resources allowed to be null and assumed to be handled on the View level (for example by Glide).
     */
    fun map(res: Resources, packageName: String, entity: BenefitsOptInEntity): BenefitsOptIn {
        var benefitsTitle = res.getIdentifier(entity.title, RESOURCE_TYPE_STRING, packageName)
        if (benefitsTitle == 0) {
            benefitsTitle =
                res.getIdentifier(benefitsTitleDefault, RESOURCE_TYPE_STRING, packageName)
        }

        val benefitsOptInItems = mutableListOf<BenefitsOptInItem>()
        for (i in entity.content) {
            var bodyRes: Int
            if (i.body.isBlank()) {
                bodyRes = res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
            } else {
                bodyRes = res.getIdentifier(i.body, RESOURCE_TYPE_STRING, packageName)
                if (bodyRes == 0) {
                    bodyRes =
                        res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
                }
            }

            val item = BenefitsOptInItem(
                res.getIdentifier(i.image, RESOURCE_TYPE_DRAWABLE, packageName),
                bodyRes
            )
            benefitsOptInItems.add(item)
        }

        var benefitsDisclaimer =
            res.getIdentifier(entity.disclaimer, RESOURCE_TYPE_STRING, packageName)
        if (benefitsDisclaimer == 0) {
            benefitsDisclaimer =
                res.getIdentifier(benefitsDisclaimerDefault, RESOURCE_TYPE_STRING, packageName)
        }
        var benefitsButtonTitle =
            res.getIdentifier(entity.buttonTitle, RESOURCE_TYPE_STRING, packageName)
        if (benefitsButtonTitle == 0) {
            benefitsButtonTitle =
                res.getIdentifier(guidedContinueDefault, RESOURCE_TYPE_STRING, packageName)
        }

        return BenefitsOptIn(
            benefitsTitle,
            benefitsOptInItems,
            benefitsDisclaimer,
            entity.disclaimerCheckBox,
            benefitsButtonTitle
        )
    }

    /**
     * Maps [LocationOptInEntity] to presentation layer [LocationOptIn] model who's data provides references to resources that are ready for binding to Views.
     * Validation on properties of [LocationOptInEntity] in this function is done to ensure that values for valid keys in guided_opt_in_config.json will always reference to existing string resources.
     * Image resources allowed to be null and assumed to be handled on the View level (for example by Glide).
     */
    fun map(res: Resources, packageName: String, entity: LocationOptInEntity): LocationOptIn {
        var locationTitle = res.getIdentifier(entity.title, RESOURCE_TYPE_STRING, packageName)
        if (locationTitle == 0) {
            locationTitle =
                res.getIdentifier(locationTitleDefault, RESOURCE_TYPE_STRING, packageName)
        }

        val locationOptInItems = mutableListOf<LocationOptInItem>()
        for (i in entity.info) {
            var bodyRes: Int
            if (i.body.isBlank()) {
                bodyRes = res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
            } else {
                bodyRes = res.getIdentifier(i.body, RESOURCE_TYPE_STRING, packageName)
                if (bodyRes == 0) {
                    bodyRes =
                        res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
                }
            }

            val item = LocationOptInItem(
                res.getIdentifier(i.image, RESOURCE_TYPE_DRAWABLE, packageName),
                bodyRes
            )
            locationOptInItems.add(item)
        }

        var locationNext = res.getIdentifier(entity.next, RESOURCE_TYPE_STRING, packageName)
        if (locationNext == 0) {
            locationNext =
                res.getIdentifier(guidedContinueDefault, RESOURCE_TYPE_STRING, packageName)
        }
        var locationSkip = res.getIdentifier(entity.skip, RESOURCE_TYPE_STRING, packageName)
        if (locationSkip == 0) {
            locationSkip = res.getIdentifier(guidedSkipDefault, RESOURCE_TYPE_STRING, packageName)
        }

        return LocationOptIn(
            entity.exclude,
            locationTitle,
            locationOptInItems,
            locationNext,
            locationSkip
        )
    }

    /**
     * Maps [NotificationsOptInEntity] to presentation layer [NotificationsOptIn] model who's data provides references to resources that are ready for binding to Views.
     * Validation on properties of [NotificationsOptInEntity] in this function is done to ensure that values for valid keys in guided_opt_in_config.json will always reference to existing string resources.
     * Image resources allowed to be null and assumed to be handled on the View level (for example by Glide).
     */
    fun map(
        res: Resources,
        packageName: String,
        entity: NotificationsOptInEntity
    ): NotificationsOptIn {
        var notificationsTitle = res.getIdentifier(entity.title, RESOURCE_TYPE_STRING, packageName)
        if (notificationsTitle == 0) {
            notificationsTitle =
                res.getIdentifier(notificationsTitleDefault, RESOURCE_TYPE_STRING, packageName)
        }

        val notificationsOptInItems = mutableListOf<NotificationsOptInItem>()
        for (i in entity.info) {
            var bodyRes: Int
            if (i.body.isBlank()) {
                bodyRes = res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
            } else {
                bodyRes = res.getIdentifier(i.body, RESOURCE_TYPE_STRING, packageName)
                if (bodyRes == 0) {
                    bodyRes =
                        res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
                }
            }

            val item = NotificationsOptInItem(
                res.getIdentifier(i.image, RESOURCE_TYPE_DRAWABLE, packageName),
                bodyRes
            )
            notificationsOptInItems.add(item)
        }

        var notificationsDisclaimerText = res.getIdentifier(entity.disclaimerText, RESOURCE_TYPE_STRING, packageName)
        if (notificationsDisclaimerText == 0) {
            notificationsDisclaimerText = res.getIdentifier(notificationsDisclaimerTextDefault, RESOURCE_TYPE_STRING, packageName)
        }

        var notificationsNext = res.getIdentifier(entity.next, RESOURCE_TYPE_STRING, packageName)
        if (notificationsNext == 0) {
            notificationsNext =
                res.getIdentifier(guidedContinueDefault, RESOURCE_TYPE_STRING, packageName)
        }
        var notificationsSkip = res.getIdentifier(entity.skip, RESOURCE_TYPE_STRING, packageName)
        if (notificationsSkip == 0) {
            notificationsSkip =
                res.getIdentifier(guidedSkipDefault, RESOURCE_TYPE_STRING, packageName)
        }

        return NotificationsOptIn(
            entity.exclude,
            notificationsTitle,
            notificationsOptInItems,
            notificationsDisclaimerText,
            notificationsNext,
            notificationsSkip
        )
    }


    /**
     * Maps [ConciergeOptOutEntity] to presentation layer [ConciergeOptOut] model who's data provides references to resources that are ready for binding to Views.
     * Validation on properties of [ConciergeOptOutEntity] in this function is done to ensure that values for valid keys in guided_opt_in_config.json will always reference to existing string resources.
     * Image resources allowed to be null and assumed to be handled on the View level (for example by Glide).
     */
    fun map(
        res: Resources,
        packageName: String,
        entity: ConciergeOptOutEntity
    ): ConciergeOptOut {
        var conciergeOptOutTitle =
            res.getIdentifier(entity.title, RESOURCE_TYPE_STRING, packageName)
        if (conciergeOptOutTitle == 0) {
            conciergeOptOutTitle =
                res.getIdentifier(conciergeOptOutTitleDefault, RESOURCE_TYPE_STRING, packageName)
        }

        val conciergeOptOutItems = mutableListOf<ConciergeOptOutItem>()
        for (i in entity.info) {
            var bodyRes: Int
            if (i.body.isBlank()) {
                bodyRes = res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
            } else {
                bodyRes = res.getIdentifier(i.body, RESOURCE_TYPE_STRING, packageName)
                if (bodyRes == 0) {
                    bodyRes =
                        res.getIdentifier(guidedEmptyString, RESOURCE_TYPE_STRING, packageName)
                }
            }

            val item = ConciergeOptOutItem(
                res.getIdentifier(i.image, RESOURCE_TYPE_DRAWABLE, packageName),
                bodyRes
            )
            conciergeOptOutItems.add(item)
        }

        var conciergeOptOutConfirm =
            res.getIdentifier(entity.confirm, RESOURCE_TYPE_STRING, packageName)
        if (conciergeOptOutConfirm == 0) {
            conciergeOptOutConfirm =
                res.getIdentifier(guidedConfirmDefault, RESOURCE_TYPE_STRING, packageName)
        }
        var conciergeOptOutCancel =
            res.getIdentifier(entity.cancel, RESOURCE_TYPE_STRING, packageName)
        if (conciergeOptOutCancel == 0) {
            conciergeOptOutCancel =
                res.getIdentifier(guidedCancelDefault, RESOURCE_TYPE_STRING, packageName)
        }

        return ConciergeOptOut(
            conciergeOptOutTitle,
            conciergeOptOutItems,
            conciergeOptOutConfirm,
            conciergeOptOutCancel
        )
    }
}