package com.flybits.concierge.services

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import com.flybits.android.push.models.newPush.ContentPush
import com.flybits.android.push.models.newPush.DisplayablePush
import com.flybits.android.push.models.newPush.EventPush
import com.flybits.android.push.models.newPush.Push
import com.flybits.android.push.services.PushService
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.InternalPreferences
import com.flybits.concierge.Utils

/**
 * Extend this class to automate the handling of Flybits push notifications.
 *
 * If push notifications are already being handled in a service in such a way that subclassing isn't possible
 * then see the methods found in the companion object for this class.
 */
const val TAG_CONCIERGE_MESSAGING = "ConciergeMessagingService"

abstract class ConciergeMessagingService(private val autoUpdate: Boolean = false) : PushService() {

    override fun onDisplayableFlybitsPushReceived(push: DisplayablePush) {
        super.onDisplayableFlybitsPushReceived(push)

        if (push is ContentPush) {
            Logger.i("ContentPush received $push")
            broadcastIntent(this, ConciergeConstants.BROADCAST_CONTENT_PUSH_RECEIVED)
        }
    }

    override fun onEntityFlybitsPushReceived(push: EventPush) {
        super.onEntityFlybitsPushReceived(push)
        Logger.i("RelevantContentPush received $push")
        if (autoUpdate) {
            broadcastIntent(this, ConciergeConstants.BROADCAST_ENTITY_PUSH_RECEIVED)
        }
    }

    override fun onNewToken(recentToken: String?) {
        super.onNewToken(recentToken)

        Logger.appendTag(TAG_CONCIERGE_MESSAGING).d("onNewToken() token: $recentToken")

        if (recentToken != null) {
            FlybitsConcierge.with(applicationContext).enablePushMessaging(recentToken)
        }
    }

    /**
     * Methods in this object are purposely made globally available to accommodate for
     * applications that are handling notifications in such a way where the [ConciergeMessagingService]
     * cannot be subclassed.
     */
    companion object {

        /**
         * Create [PendingIntent] for the associated [Push].
         *
         * @param context Context of the application.
         * @param push [Push] that will be used in constructing the PendingIntent.

         * @return The created [PendingIntent].
         */
        @Throws(FlybitsException::class)
        fun createPendingIntent(
            context: Context,
            push: DisplayablePush,
            intent: Intent
        ): PendingIntent {
            return PendingIntent.getActivity(
                context
                , push.id.hashCode(), intent, PendingIntent.FLAG_UPDATE_CURRENT
            )
        }

        /**
         * Create [Intent] which will contain the launcher activity and the passed [Push].
         *
         * @param context Context of the application.
         * @param push [Push] that will be inserted into the Intent.
         *
         * @return The created [Intent].
         */
        @Throws(FlybitsException::class)
        fun createIntent(context: Context, push: DisplayablePush): Intent {
            val activityClass = Utils.launcherActivity(context)
                ?: throw FlybitsException("No launcher activity found!")
            return Intent(context, activityClass).apply {
                flags = Intent.FLAG_ACTIVITY_CLEAR_TOP or
                        Intent.FLAG_ACTIVITY_SINGLE_TOP or
                        Intent.FLAG_ACTIVITY_NEW_TASK
                putExtra(ConciergeConstants.PUSH_EXTRA, push)
            }
        }

        /**
         * Save the push token locally.
         *
         * @param context Context of the application.
         */
        @JvmStatic
        fun savePushToken(recentToken: String, context: Context) {
            InternalPreferences.savePushToken(context, recentToken)
        }

        /**
         * Send broadcast with custom action
         *
         * @param context Context of the application.
         * @param action the action of the broadcast to be sent
         */
        fun broadcastIntent(context: Context, action: String) {
            Intent().also { intent ->
                intent.action = action
                context.sendBroadcast(intent)
            }
        }
    }
}
